﻿<?php

header("Content-Type: application/json; charset=UTF-8");

// Define o caminho para o arquivo do token
$tokenFile = 'token.json';

/**
 * Função para consultar uma API remota usando cURL.
 *
 * @param string $url O URL do endpoint da API.
 * @param array $headers Um array de cabeçalhos HTTP para enviar com a requisição.
 * @return array Um array associativo com os detalhes da resposta.
 */
function consultarAPI($url, $headers) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_HEADER => true,
        CURLOPT_TIMEOUT => 60
    ]);

    $response = curl_exec($ch);
    $info = curl_getinfo($ch);
    $error = curl_error($ch);
    curl_close($ch);

    return [
        "rota" => $url,
        "status" => $info["http_code"] ?? 0,
        "online" => ($info["http_code"] >= 200 && $info["http_code"] < 500),
        "headers" => substr($response, 0, $info["header_size"] ?? 0),
        "body" => substr($response, $info["header_size"] ?? 0),
        "erroCurl" => $error
    ];
}

// ============================
// GESTÃO DO TOKEN
// ============================
$bearerToken = "";
$headers = [];

// Verifica se um novo token foi passado via requisição GET
if (isset($_REQUEST['bearer'])) {
    $bearerToken = 'Bearer ' . filter_var($_REQUEST['bearer'], FILTER_SANITIZE_SPECIAL_CHARS);
    
    // Cria ou atualiza o arquivo token.json com o novo token
    $tokenData = [
        'token' => $bearerToken,
        'created_at' => date('Y-m-d H:i:s'),
        'expires_in' => '' // Este valor precisa ser obtido do payload do token
    ];
    file_put_contents($tokenFile, json_encode($tokenData, JSON_PRETTY_PRINT));
} else {
    // Lê o token do arquivo JSON
    if (file_exists($tokenFile)) {
        $tokenData = json_decode(file_get_contents($tokenFile), true);
        if (is_array($tokenData) && isset($tokenData['token'])) {
            $bearerToken = $tokenData['token'];
        }
    }
    // Se não houver token, retorna erro
    if (empty($bearerToken)) {
        echo json_encode([
            'error' => 'Nenhum token de autenticação fornecido ou encontrado.'
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }
}

// ============================
// CABEÇALHOS FIXOS (com o token)
// ============================
$headers = [
    "Accept: application/json, text/plain, */*",
    "Accept-Encoding: gzip, deflate, br, zstd",
    "Accept-Language: pt-BR,pt;q=0.9,en-US;q=0.8,en;q=0.7",
    "Authorization: " . $bearerToken,
    "Connection: keep-alive",
    "Referer: https://portaldeservicos.pdpj.jus.br/consulta",
    "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64)"
];

// ============================
// VALORES VINDO VIA GET/POST
// ============================
$oab = filter_var($_REQUEST["oab"] ?? "", FILTER_SANITIZE_SPECIAL_CHARS);
$processo = filter_var($_REQUEST["processo"] ?? "", FILTER_SANITIZE_SPECIAL_CHARS);
$cpf = filter_var($_REQUEST["cpf"] ?? "", FILTER_SANITIZE_SPECIAL_CHARS);

// ============================
// ROTAS BASE
// ============================
$rotas = [
    "Consulta por OAB" => $oab ? "https://portaldeservicos.pdpj.jus.br/api/v2/processos?oabRepresentante=" . urlencode($oab) : "",
    "Consulta por Processo" => $processo ? "https://portaldeservicos.pdpj.jus.br/api/v2/processos/" . $processo : "",
    "Consulta por CPF/CNPJ" => $cpf ? "https://portaldeservicos.pdpj.jus.br/api/v2/processos?cpfCnpjParte=" . urlencode($cpf) : ""
];

// ============================
// EXECUTAR CONSULTAS E REFORMATAR A RESPOSTA
// ============================
$resultados = [];

// Determina qual consulta foi solicitada
$consultaSolicitada = "";
if (!empty($oab)) {
    $consultaSolicitada = "Consulta por OAB";
} elseif (!empty($processo)) {
    $consultaSolicitada = "Consulta por Processo";
} elseif (!empty($cpf)) {
    $consultaSolicitada = "Consulta por CPF/CNPJ";
}

// Executa apenas a consulta solicitada
if ($consultaSolicitada && !empty($rotas[$consultaSolicitada])) {
    $url = $rotas[$consultaSolicitada];
    $apiResponse = consultarAPI($url, $headers);
    
    // Verifica erro no cURL
    if (!empty($apiResponse['erroCurl']) || !$apiResponse['online']) {
        $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Erro na requisição: " . $apiResponse['erroCurl']]);
    } elseif ($consultaSolicitada === "Consulta por Processo" && !empty($apiResponse["body"])) {
        // Decodifica o JSON da resposta
        $processoData = json_decode($apiResponse["body"], true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $resultados[$consultaSolicitada] = array_merge($apiResponse, ["json_error" => json_last_error_msg()]);
        } elseif (is_array($processoData) && !empty($processoData) && isset($processoData[0])) {
            // Supõe que a resposta é um array com um único objeto
            $p = $processoData[0];
            $poloAtivo = [];
            $poloPassivo = [];
            $advogados = []; // Novo array para armazenar advogados

            // CORREÇÃO: Acessar partes dentro de tramitacaoAtual
            if (isset($p['tramitacaoAtual']['partes']) && is_array($p['tramitacaoAtual']['partes'])) {
                foreach ($p['tramitacaoAtual']['partes'] as $parte) {
                    if (!isset($parte['polo'], $parte['nome'])) continue;

                    $nomeParte = $parte['nome'];
                    $documentoParte = '';

                    // Extrai CPF/CNPJ da parte
                    if (isset($parte['documentosPrincipais']) && is_array($parte['documentosPrincipais'])) {
                        foreach ($parte['documentosPrincipais'] as $doc) {
                            if (in_array($doc['tipo'], ['CPF', 'CNPJ'])) {
                                $documentoParte = $doc['numero'];
                                break;
                            }
                        }
                    }

                    $stringParte = $nomeParte . ($documentoParte ? " (Doc: $documentoParte)" : "");

                    // Agrupa por polo
                    if ($parte['polo'] === 'ATIVO') {
                        $poloAtivo[] = $stringParte;
                        
                        // NOVO: Adiciona a lógica para extrair advogados do polo ativo
                        if (isset($parte['representantes']) && is_array($parte['representantes'])) {
                            foreach ($parte['representantes'] as $representante) {
                                if (isset($representante['tipoRepresentacao']) && $representante['tipoRepresentacao'] === 'ADVOGADO') {
                                    $nomeAdv = $representante['nome'] ?? 'N/A';
                                    $cpfAdv = 'N/A';
                                    if (isset($representante['cadastroReceitaFederal']) && is_array($representante['cadastroReceitaFederal'])) {
                                        foreach ($representante['cadastroReceitaFederal'] as $docAdv) {
                                            if ($docAdv['tipo'] === 'CPF') {
                                                $cpfAdv = $docAdv['numero'];
                                                break;
                                            }
                                        }
                                    }
                                    // Adiciona o advogado ao array
                                    $advogados[] = "Advogado: $nomeAdv" . ($cpfAdv !== 'N/A' ? " (CPF: $cpfAdv)" : "");
                                }
                            }
                        }

                    } elseif ($parte['polo'] === 'PASSIVO') {
                        $poloPassivo[] = $stringParte;
                    }
                }
            }

            // Junta as múltiplas partes do mesmo polo
            $poloAtivoStr = !empty($poloAtivo) ? implode('; ', $poloAtivo) : 'N/A';
            $poloPassivoStr = !empty($poloPassivo) ? implode('; ', $poloPassivo) : 'N/A';
            $advogadosStr = !empty($advogados) ? implode('; ', $advogados) : 'N/A';

            // NOVO: Refatora a resposta para a nova estrutura
            $reformattedData = [
                "Detalhes do Processo" => [
                    "Número do Processo" => $p['numeroProcesso'] ?? 'N/A',
                    "Tribunal" => $p['tramitacaoAtual']['tribunal']['nome'] ?? 'N/A',
                    "Órgão Julgador" => $p['tramitacaoAtual']['distribuicao'][0]['orgaoJulgador'][0]['nome'] ?? 'N/A',
                    "Classe" => $p['tramitacaoAtual']['classe'][0]['descricao'] ?? 'N/A',
                    "Assunto" => $p['tramitacaoAtual']['assunto'][0]['descricao'] ?? 'N/A',
                    "Valor da Ação" => $p['tramitacaoAtual']['valorAcao'] ?? 'N/A',
                    "Data do Ajuizamento" => $p['tramitacaoAtual']['dataHoraAjuizamento'] ?? 'N/A'
                ],
                "Polo Ativo" => [
                    "Parte" => $poloAtivoStr,
                    "Advogado" => $advogadosStr
                ],
                "Polo Passivo" => [
                    "Parte" => $poloPassivoStr
                ]
            ];

            // Combina os dados processados na resposta
            $resultados[$consultaSolicitada] = [
                "Dados Processados" => $reformattedData,
            ];
        } else {
            // Se os dados estiverem vazios, mantém o corpo original da resposta
            $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Resposta vazia ou inválida"]);
        }
    } else {
        // Para outras consultas, mantém o formato original da resposta
        $resultados[$consultaSolicitada] = $apiResponse;
    }
}

// ============================
// RESPOSTA JSON
// ============================
$finalResponse = ['response' => $resultados[$consultaSolicitada] ?? ['error' => 'Nenhuma consulta válida fornecida']];
echo json_encode($finalResponse, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

?>