﻿const express = require('express');
const { Client, LocalAuth, MessageMedia } = require('whatsapp-web.js');
const qrcode = require('qrcode');
const http = require('http');
const { Server } = require("socket.io");
const fs = require('fs');
const path = require('path');
const axios = require('axios');
const archiver = require('archiver');
const app = express();
const server = http.createServer(app);
const io = new Server(server);
const PORT = process.env.PORT || 1111;
const sessions = new Map();
const MAX_SESSIONS = 200;
const MAX_PDFS = 1000;
const SESSIONS_FILE = './211sessions.json';
const gerarFlowState = new Map();
const gerarFlowData = new Map();
const oabFlowState = new Map();
const oabFlowData = new Map();
process.setMaxListeners(0);
process.on('uncaughtException', (err) => {
    console.error('Exceção não tratada capturada:', err.message, err.stack);
});
process.on('unhandledRejection', (reason, promise) => {
    console.error('Rejeição não tratada:', reason);
});
function isValidClientId(clientId) {
    return /^[a-zA-Z0-9_-]+$/.test(clientId);
}
function cleanHtmlTags(str) {
    if (typeof str !== 'string') return str;
    return str.replace(/<[^>]*>?/gm, '').trim();
}
const fetchWithRetry = async (url, retries = 3, delay = 2000) => {
    for (let i = 0; i < retries; i++) {
        try {
            console.log(`[${new Date().toLocaleTimeString()}] Tentativa ${i + 1}/${retries} para: ${url.substring(0, 50)}...`);
            const response = await axios.get(url, {
                responseType: 'arraybuffer',
                timeout: 15000,
                headers: {
                    'User-Agent': 'WhatsAppBot/1.0',
                    'Accept': 'application/pdf'
                }
            });
            if (response.status === 200 && response.headers['content-type']?.includes('pdf')) {
                console.log(`✅ Sucesso na tentativa ${i + 1} - ${response.headers['content-length']} bytes recebidos`);
                return response;
            } else {
                throw new Error(`Status ${response.status}: ${response.statusText}`);
            }
        } catch (e) {
            console.error(`Falha na requisição (tentativa ${i + 1}/${retries}): ${e.message}`);
            if (i < retries - 1) {
                await new Promise(res => setTimeout(res, delay * Math.pow(2, i)));
            }
        }
    }
    throw new Error('Falha na requisição após múltiplas tentativas.');
};
function numeroPorExtenso(num) {
    if (num === 0) return 'zero';
    const unidades = ['', 'um', 'dois', 'três', 'quatro', 'cinco', 'seis', 'sete', 'oito', 'nove'];
    const dezenas = ['', 'dez', 'vinte', 'trinta', 'quarenta', 'cinquenta', 'sessenta', 'setenta', 'oitenta', 'noventa'];
    const centenas = ['', 'cento', 'duzentos', 'trezentos', 'quatrocentos', 'quinhentos', 'seiscentos', 'setecentos', 'oitocentos', 'novecentos'];
    const especiais = ['dez', 'onze', 'doze', 'treze', 'quatorze', 'quinze', 'dezesseis', 'dezessete', 'dezoito', 'dezenove'];
    let str = '';
    if (num >= 1000) {
        const mil = Math.floor(num / 1000);
        str += numeroPorExtenso(mil) + ' mil ';
        num %= 1000;
    }
    if (num >= 100) {
        const cent = Math.floor(num / 100);
        str += centenas[cent] + ' ';
        num %= 100;
    }
    if (num >= 20) {
        const dez = Math.floor(num / 10);
        str += dezenas[dez] + ' ';
        num %= 10;
    } else if (num >= 10) {
        str += especiais[num - 10] + ' ';
        num = 0;
    }
    if (num > 0) str += unidades[num] + ' ';
    return str.trim();
}
function valorPorExtenso(valorStr) {
    let valorClean = valorStr.replace('R$ ', '').replace(/\./g, '').replace(',', '.').trim();
    const valor = parseFloat(valorClean);
    const inteiro = Math.floor(valor);
    const decimal = Math.round((valor - inteiro) * 100);
    let extenso = numeroPorExtenso(inteiro) + ' reais';
    if (decimal > 0) {
        extenso += ' e ' + numeroPorExtenso(decimal) + ' centavos';
    }
    return extenso;
}
async function getCpfData(cpf) {
    if (!cpf || cpf.length !== 11) return null;
    const apiKey = '6fc9a1c8-f4fe-4106-b9ce-d2b342ae50ed';
    const url = `http://191.96.224.215/maumau.php?cpf=${cpf}`;
    try {
        const response = await axios.get(url);
        if (response.data.Status === 'success') {
            return response.data;
        }
    } catch (e) {
        console.error(`Erro ao consultar CPF ${cpf}: ${e.message}`);
    }
    return null;
}
function calcularIdade(nascStr) {
    if (!nascStr) return 'N/A';
    const nasc = new Date(nascStr);
    const hoje = new Date();
    let idade = hoje.getFullYear() - nasc.getFullYear();
    const mes = hoje.getMonth() - nasc.getMonth();
    if (mes < 0 || (mes === 0 && hoje.getDate() < nasc.getDate())) idade--;
    return idade;
}
function saveSessions() {
    const sessionIds = Array.from(sessions.keys());
    fs.writeFileSync(SESSIONS_FILE, JSON.stringify(sessionIds), 'utf8');
}
function loadSessions() {
    if (fs.existsSync(SESSIONS_FILE)) {
        const sessionIds = JSON.parse(fs.readFileSync(SESSIONS_FILE, 'utf8'));
        sessionIds.forEach(id => {
            if (isValidClientId(id) && !sessions.has(id)) {
                sessions.set(id, { clientId: id, status: 'Desconectado', connected: false, client: null });
            } else if (!isValidClientId(id)) {
                console.warn(`ID de sessão inválido ignorado ao carregar: ${id}`);
            }
        });
        console.log(`Sessões carregadas: ${sessionIds.join(', ')}`);
    }
}
function broadcastSessionsUpdate() {
    const sessionStates = {};
    sessions.forEach((s, id) => {
        sessionStates[id] = { clientId: id, status: s.status, qrCodeUrl: s.qrCodeUrl, connected: s.connected };
    });
    io.emit('sessions_update', sessionStates);
}
function logToClient(clientId, message) {
    io.emit('log_update', { clientId, message, timestamp: new Date().toLocaleTimeString() });
}
async function destroySession(clientId, removeFromDisk = false) {
    const s = sessions.get(clientId);
    if (!s) return;
    try {
        if (s.client && s.client.pupBrowser) {
            await s.client.pupBrowser.close();
        }
        if (s.client) await s.client.destroy();
    } catch (e) {
        console.error(`Erro ao destruir o cliente da sessão ${clientId}:`, e.message);
    } finally {
        s.client = null;
        s.connected = false;
        s.status = "Desconectado";
        s.qrCodeUrl = '';
        if (removeFromDisk) {
            const authDir = path.join(__dirname, '111.wwebjs_auth', `session-${clientId}`);
            if (fs.existsSync(authDir)) {
                try {
                    fs.rmSync(authDir, { recursive: true, force: true });
                    console.log(`Diretório de autenticação ${clientId} removido com sucesso.`);
                } catch (e) {
                    console.error(`Erro ao remover o diretório de autenticação ${clientId}:`, e.message);
                }
            }
            sessions.delete(clientId);
        }
        broadcastSessionsUpdate();
        saveSessions();
    }
}
async function initializeSession(clientId) {
    if (!isValidClientId(clientId)) {
        console.error(`ID de sessão inválido: ${clientId}. Ignorando inicialização.`);
        return;
    }
    if (sessions.size >= MAX_SESSIONS && !sessions.has(clientId)) {
        console.log("⚠️ Limite de sessões atingido.");
        return;
    }
    let s = sessions.get(clientId);
    if (!s) {
        sessions.set(clientId, { clientId, status: 'Novo', connected: false, client: null });
        s = sessions.get(clientId);
    }
    if (s.connected || s.client) {
        console.log(`Sessão ${clientId} já está conectada ou em processo de inicialização.`);
        return;
    }
    s.status = "Inicializando...";
    broadcastSessionsUpdate();
    logToClient(clientId, "Iniciando sessão...");
    try {
        const client = new Client({
            authStrategy: new LocalAuth({ clientId }),
            puppeteer: {
                headless: true,
                args: ['--no-sandbox', '--disable-setuid-sandbox', '--disable-gpu', '--disable-software-rasterizer', '--no-zygote', '--single-process', '--disable-dev-shm-usage'],
                protocolTimeout: 0
            }
        });
        s.client = client;
        client.on('qr', async qr => {
            if (sessions.has(clientId)) {
                sessions.get(clientId).qrCodeUrl = await qrcode.toDataURL(qr);
                sessions.get(clientId).status = "Aguardando QR Code";
                broadcastSessionsUpdate();
                logToClient(clientId, "QR Code gerado. Escaneie para conectar.");
            }
        });
        client.on('ready', () => {
            if (sessions.has(clientId)) {
                const ss = sessions.get(clientId);
                ss.status = "Conectado";
                ss.qrCodeUrl = '';
                ss.connected = true;
                broadcastSessionsUpdate();
                logToClient(clientId, "Sessão conectada com sucesso!");
            }
        });
        client.on('disconnected', (reason) => {
            logToClient(clientId, `Sessão desconectada. Motivo: ${reason}`);
            destroySession(clientId, false);
        });
        client.on('auth_failure', (reason) => {
            logToClient(clientId, `Falha de autenticação. Motivo: ${reason}`);
            destroySession(clientId, true);
        });
        client.on('message', async msg => {
            if (!msg || !msg.from) {
                console.error(`[${clientId}] Mensagem inválida recebida sem remetente.`);
                logToClient(clientId, 'Mensagem inválida sem remetente.');
                return;
            }
            const from = msg.from; // Garantindo que 'from' esteja sempre definido aqui
            try {
                const lowerCaseInput = msg.body.toLowerCase().trim();
                const userInput = msg.body.trim();
                if (lowerCaseInput === '/menu') {
                    const menuMessage = `
*Comandos disponíveis:*
/menu - Exibe este menu
/oab [UF+NUMERO] - Consulta processos por OAB (ex: /oab SP450197)
/oabpdf [UF+NUMERO] [NUMERO] - Gera PDFs para os processos mais recentes e de maior valor (ex: /oabpdf SP450197 5)
/oabzip [UF+NUMERO] [NUMERO] - Gera ZIP com PDFs, detalhes e script (ex: /oabzip SP450197 5)
/nome [NOME COMPLETO] - Consulta dados de advogado por nome
/processo [NÚMERO DO PROCESSO] - Consulta detalhes de um processo
/pdf [NÚMERO DO PROCESSO] - Gera um documento PDF para o processo
/cpf [CPF] - Consulta dados de CPF (ex: /cpf 12345678901)
`;
                    await msg.reply(menuMessage);
                    return;
                }
                if (lowerCaseInput === '/gerar') {
                    gerarFlowState.set(from, 'ask_name');
                    gerarFlowData.set(from, {});
                    await msg.reply('Olá! Para gerar o documento, vamos precisar de algumas informações. Por favor, digite o seu nome completo:');
                    return;
                }
                if (lowerCaseInput.startsWith('/oabpdf')) {
                    const parts = lowerCaseInput.split(' ');
                    const rawOab = parts[1];
                    const numPdfs = parts[2] ? parseInt(parts[2]) : null;
                    if (rawOab) {
                        const state = rawOab.substring(0, 2).toUpperCase();
                        const number = rawOab.substring(2).replace(/\D/g, '');
                        const formattedOab = state + number;
                        await handleOabPdfGeneration(msg, formattedOab, numPdfs, client, clientId);
                    } else {
                        await msg.reply('Uso: /oabpdf [UF+NUMERO] [QUANTIDADE] (ex: /oabpdf SP450197 5)');
                    }
                } else if (lowerCaseInput.startsWith('/oabzip')) {
                    const parts = lowerCaseInput.split(' ');
                    const rawOab = parts[1];
                    const numPdfs = parts[2] ? parseInt(parts[2]) : null;
                    if (rawOab) {
                        const state = rawOab.substring(0, 2).toUpperCase();
                        const number = rawOab.substring(2).replace(/\D/g, '');
                        const formattedOab = state + number;
                        await handleOabZipGeneration(msg, formattedOab, numPdfs, client, clientId);
                    } else {
                        await msg.reply('Uso: /oabzip [UF+NUMERO] [QUANTIDADE] (ex: /oabzip SP450197 5)');
                    }
                } else if (lowerCaseInput.startsWith('/oab')) {
                    const rawOab = lowerCaseInput.split(' ')[1];
                    if (rawOab) {
                        const state = rawOab.substring(0, 2).toUpperCase();
                        const number = rawOab.substring(2).replace(/\D/g, '');
                        const formattedOab = state + number;
                        await handleOabConsultaByNumber(msg, formattedOab, client, clientId);
                    } else {
                        await msg.reply('Uso: /oab [UF+NUMERO] (ex: /oab SP450197)');
                    }
                } else if (lowerCaseInput.startsWith('/nome')) {
                    const nomeCompleto = userInput.substring(userInput.indexOf(' ') + 1);
                    if (nomeCompleto) {
                        await handleOabConsultaByName(msg, nomeCompleto, client, clientId);
                    } else {
                        await msg.reply('Por favor, digite o nome completo para a consulta. Ex: /nome Marcos Godoy');
                    }
                } else if (lowerCaseInput.startsWith('/processo')) {
                    const numeroProcesso = userInput.substring(userInput.indexOf(' ') + 1);
                    if (numeroProcesso) {
                        await handleProcessoConsulta(msg, numeroProcesso, client, clientId);
                    } else {
                        await msg.reply('Uso: /processo [numero_do_processo] (ex: /processo 5001879-75.2025.8.13.0540)');
                    }
                } else if (lowerCaseInput.startsWith('/pdf')) {
                    const numeroProcesso = userInput.substring(userInput.indexOf(' ') + 1);
                    if (numeroProcesso) {
                        await handlePdfGenerationFlowStart(msg, numeroProcesso, client, clientId);
                    } else {
                        await msg.reply('Uso: /pdf [numero_do_processo] para gerar um documento PDF (ex: /pdf 5001879-75.2025.8.13.0540)');
                    }
                } else if (lowerCaseInput.startsWith('/cpf')) {
                    const cpf = userInput.substring(userInput.indexOf(' ') + 1).replace(/\D/g, '');
                    if (cpf.length === 11) {
                        await handleCpfConsulta(msg, cpf, client, clientId);
                    } else {
                        await msg.reply('Uso: /cpf [CPF] (ex: /cpf 12345678901)');
                    }
                }
                if (gerarFlowState.has(from)) {
                    const currentStep = gerarFlowState.get(from);
                    let flowData = gerarFlowData.get(from) || {};
                    switch (currentStep) {
                        case 'ask_name':
                            flowData.parte1_nome = userInput;
                            gerarFlowState.set(from, 'ask_cpf');
                            await msg.reply('Certo. Agora, por favor, digite o CPF (apenas os números).');
                            break;
                        case 'ask_cpf':
                            const cpf = userInput.replace(/\D/g, '');
                            if (cpf.length === 11) {
                                flowData.parte1_cpf = cpf;
                                gerarFlowState.set(from, 'ask_advogado');
                                await msg.reply('Entendido. Qual o nome do advogado(a)?');
                            } else {
                                await msg.reply('CPF inválido. Por favor, digite 11 dígitos.');
                            }
                            break;
                        case 'ask_advogado':
                            flowData.parte1_advogado = userInput;
                            gerarFlowState.set(from, 'ask_processo');
                            await msg.reply('Qual o número do processo?');
                            break;
                        case 'ask_processo':
                            flowData.processo = userInput;
                            gerarFlowState.set(from, 'ask_parte2');
                            await msg.reply('Quem é a parte contrária?');
                            break;
                        case 'ask_parte2':
                            flowData.parte2_nome = userInput;
                            gerarFlowState.set(from, 'ask_valor');
                            await msg.reply('Qual o valor a ser recebido? (ex: 1.234,56)');
                            break;
                        case 'ask_valor':
                            flowData.valor_a_receber = userInput;
                            gerarFlowData.set(from, flowData);
                            gerarFlowState.set(from, 'confirm_data');
                            await sendSummaryMessage(from, flowData, client);
                            break;
                        case 'ask_missing_info':
                            flowData.parte1_advogado = userInput;
                            gerarFlowState.set(from, 'confirm_data');
                            await sendSummaryMessage(from, flowData, client, true);
                            break;
                        case 'confirm_data':
                            if (['sim', 's', 'yes', 'y', 'ok'].includes(lowerCaseInput)) {
                                await generateAndSendPdf(from, flowData, client, clientId);
                            } else if (['nao', 'não', 'n', 'no'].includes(lowerCaseInput)) {
                                gerarFlowState.set(from, 'ask_correction_field');
                                const correctionMessage = [
                                    'Qual campo deseja corrigir?',
                                    `*1. Nome*: ${flowData.parte1_nome}`,
                                    `*2. CPF*: ${flowData.parte1_cpf}`,
                                    `*3. Advogado*: ${flowData.parte1_advogado}`,
                                    `*4. Processo*: ${flowData.processo}`,
                                    `*5. Parte Contrária*: ${flowData.parte2_nome}`,
                                    `*6. Valor*: ${flowData.valor_a_receber}`
                                ].join('\n');
                                await msg.reply(correctionMessage);
                            } else {
                                await msg.reply('Resposta inválida. Por favor, responda com *Sim* ou *Não*.');
                            }
                            break;
                        case 'ask_correction_field':
                            const fieldNumber = parseInt(userInput, 10);
                            const fields = ['parte1_nome', 'parte1_cpf', 'parte1_advogado', 'processo', 'parte2_nome', 'valor_a_receber'];
                            if (fieldNumber >= 1 && fieldNumber <= fields.length) {
                                gerarFlowState.set(from, `correcting_${fieldNumber}`);
                                const fieldName = fields[fieldNumber - 1].replace(/_/g, ' ');
                                await msg.reply(`Por favor, digite o novo valor para *${fieldName.charAt(0).toUpperCase() + fieldName.slice(1)}*.`);
                            } else {
                                await msg.reply('Número de campo inválido. Por favor, digite o número do campo que deseja corrigir.');
                            }
                            break;
                        case 'correcting_1': flowData.parte1_nome = userInput; gerarFlowState.set(from, 'confirm_data'); await sendSummaryMessage(from, flowData, client, true); break;
                        case 'correcting_2': flowData.parte1_cpf = userInput.replace(/\D/g, ''); gerarFlowState.set(from, 'confirm_data'); await sendSummaryMessage(from, flowData, client, true); break;
                        case 'correcting_3': flowData.parte1_advogado = userInput; gerarFlowState.set(from, 'confirm_data'); await sendSummaryMessage(from, flowData, client, true); break;
                        case 'correcting_4': flowData.processo = userInput; gerarFlowState.set(from, 'confirm_data'); await sendSummaryMessage(from, flowData, client, true); break;
                        case 'correcting_5': flowData.parte2_nome = userInput; gerarFlowState.set(from, 'confirm_data'); await sendSummaryMessage(from, flowData, client, true); break;
                        case 'correcting_6': flowData.valor_a_receber = userInput; gerarFlowState.set(from, 'confirm_data'); await sendSummaryMessage(from, flowData, client, true); break;
                        default:
                            gerarFlowState.delete(from);
                            gerarFlowData.delete(from);
                            await msg.reply('Ocorreu um erro no fluxo de geração. Por favor, comece novamente com o comando */gerar*.');
                            break;
                    }
                    return;
                } else if (oabFlowState.has(from)) {
                    const currentStep = oabFlowState.get(from);
                    const flowData = oabFlowData.get(from);
                    switch (currentStep) {
                        case 'ask_num_pdfs':
                            const numPdfs = parseInt(userInput.trim());
                            if (isNaN(numPdfs) || numPdfs <= 0 || numPdfs > flowData.maxPdfs) {
                                await msg.reply(`Por favor, informe um número válido de PDFs (1 a ${flowData.maxPdfs}).`);
                                return;
                            }
                            oabFlowState.delete(from);
                            if (flowData.command === 'oabzip') {
                                await handleOabZipGeneration(msg, flowData.formattedOab, numPdfs, client, clientId, flowData.oabData);
                            } else if (flowData.command === 'oabpdf') {
                                await handleOabPdfGeneration(msg, flowData.formattedOab, numPdfs, client, clientId, flowData.oabData);
                            }
                            oabFlowData.delete(from);
                            break;
                    }
                    return;
                }
            } catch (err) {
                console.error(`[${clientId}] Erro no manipulador de mensagens para ${from || 'unknown'}:`, err.message, err.stack);
                logToClient(clientId, `Erro no manipulador de mensagens: ${err.message}`);
                try {
                    await msg.reply('❌ Ocorreu um erro interno ao processar sua solicitação. Tente novamente ou contate o suporte.');
                } catch (sendErr) {
                    console.error(`[${clientId}] Falha ao enviar resposta de erro:`, sendErr.message);
                }
            }
        });
        await client.initialize();
    } catch (err) {
        console.error(`Falha ao inicializar a sessão ${clientId}:`, err.message);
        logToClient(clientId, `Erro na inicialização: ${err.message}. Tentando reiniciar...`);
        s.status = "Erro na Inicialização";
        broadcastSessionsUpdate();
        setTimeout(() => {
            if (!s.connected) {
                logToClient(clientId, "Reiniciando sessão automaticamente...");
                initializeSession(clientId);
            }
        }, 5000);
    }
}
async function fetchOabProcesses(formattedOab, clientId) {
    try {
        const response = await axios.get(`http://127.0.0.1:5000/consulta?oab=${formattedOab}`);
        const data = response.data;
        return data.response || {};
    } catch (e) {
        console.error(`[${clientId}] Erro ao buscar processos da OAB ${formattedOab}:`, e.message);
    }
    return {};
}
async function fetchProcessDetails(numeroProcesso, clientId) {
    try {
        const response = await axios.get(`http://127.0.0.1:5000/consulta?processo=${numeroProcesso}`);
        const data = response.data;
        return data.response?.["Dados Processados"];
    } catch (e) {
        console.error(`[${clientId}] Erro ao buscar detalhes do processo ${numeroProcesso}:`, e.message);
    }
    return null;
}
function extractDocFromParte(parteStr) {
    const match = parteStr.match(/Doc:\s*(\d+)/);
    return match ? match[1] : null;
}
function formatCpfData(cpfData) {
    if (!cpfData) return '';
    const dados = cpfData.Dados;
    const idade = calcularIdade(dados.NASC);
    const phones = cpfData.Telefones.filter(t => t).join(', ') || 'N/A';
    const renda = dados.RENDA ? `R$ ${dados.RENDA}` : 'N/A';
    return `Renda: ${renda}, Idade: ${idade}, Telefones: ${phones}`;
}
function formatCpfDataForScript(cpfData, cpf) {
    return {};
}
async function handleCpfConsulta(msg, cpf, client, clientId) {
    logToClient(clientId, `Comando /cpf ativado por ${msg.from} para o CPF ${cpf}`);
    const cpfData = await getCpfData(cpf);
    if (!cpfData) {
        await msg.reply(`⚠️ Nenhum dado encontrado para o CPF: *${cpf}*.`);
        return;
    }
    const dados = cpfData.Dados;
    const idade = calcularIdade(dados.NASC);
    const sexo = dados.SEXO === 'M' ? 'Masculino' : 'Feminino';
    const phones = cpfData.Telefones.filter(t => t).join(', ') || 'N/A';
    const renda = dados.RENDA ? `R$ ${dados.RENDA}` : 'N/A';
    const textResponse = `
*Consulta CPF: ${cpf}*
Nome: ${dados.NOME}
Sexo: ${sexo}
Data de Nascimento: ${new Date(dados.NASC).toLocaleDateString('pt-BR')}
Idade: ${idade}
Renda: ${renda}
Telefones: ${phones}
Score: ${cpfData.Score || 'N/A'}
`;
    await msg.reply(textResponse.trim());
}
async function handleOabConsultaByNumber(msg, formattedOab, client, clientId) {
    logToClient(clientId, `Comando /oab ativado por ${msg.from} para a OAB ${formattedOab}`);
    try {
        const oabData = await fetchOabProcesses(formattedOab, clientId);
        const processes = oabData.Processos || [];
        const totalProcesses = processes.length;
        if (totalProcesses === 0) {
            await msg.reply(`⚠️ Nenhum processo encontrado para a OAB: *${formattedOab}*.`);
            return;
        }
        await msg.reply(`⌛ Encontrados *${totalProcesses}* processos. Processando detalhes...`);
        let detailsTextResponse = `*Consulta Detalhada OAB: ${formattedOab}*\nTotal de processos: ${totalProcesses}\n\n`;
        let scriptTextResponse = `*Script OAB: ${formattedOab}*\nTotal de processos: ${totalProcesses}\n\n`;
        for (const process of processes) {
            const detalhes = process["Detalhes do Processo"];
            const poloAtivo = process["Polo Ativo"];
            const poloPassivo = process["Polo Passivo"] || {};
            if (!detalhes || !poloAtivo) continue;
            const valorAcao = detalhes["Valor da Ação"] || 'N/A';
            const ativoParteStr = poloAtivo.Parte || '';
            const ativoCpf = extractDocFromParte(ativoParteStr);
            const passivoParteStr = poloPassivo.Parte || '';
            const passivoDoc = extractDocFromParte(passivoParteStr);
            const ativoCpfData = await getCpfData(ativoCpf);
            const passivoCpfData = passivoDoc && passivoDoc.length === 11 ? await getCpfData(passivoDoc) : null;
            detailsTextResponse += `
-----------------------------------------
*Processo:* ${detalhes["Número do Processo"] || 'N/A'}
*Tribunal:* ${detalhes.Tribunal || 'N/A'}
*Órgão Julgador:* ${detalhes["Órgão Julgador"] || 'N/A'}
*Classe*: ${detalhes.Classe || 'N/A'}
*Assunto*: ${detalhes.Assunto || 'N/A'}
*Valor da Ação*: ${valorAcao}
*Data do Último Movimento*: ${detalhes["Data do Último Movimento"] || 'N/A'}
*Polo Ativo (Autor/Requerente):*
${ativoParteStr}
${ativoCpfData ? formatCpfData(ativoCpfData) : ''}
${poloAtivo.Advogado ? `Advogado: ${poloAtivo.Advogado}` : ''}
*Polo Passivo (Réu/Requerido):*
${passivoParteStr}
${passivoCpfData ? formatCpfData(passivoCpfData) : ''}
${poloPassivo.Advogado ? `Advogado: ${poloPassivo.Advogado}` : ''}
-----------------------------------------
`;
            const ativoNome = ativoParteStr.split('(')[0]?.trim() || 'N/A';
            const ativoAdvogado = poloAtivo.Advogado?.match(/Advogado:\s*(.+?)\s*(?:\(CPF:|$)/)?.[1]?.trim() || 'N/A';
            const passivoNome = passivoParteStr.split('(')[0]?.trim() || 'N/A';
            const passivoIcon = passivoDoc && passivoDoc.length === 14 ? '🏢' : '👤';
            const passivoDocType = passivoDoc && passivoDoc.length === 14 ? 'CNPJ' : 'CPF';
            const valorCausa = valorAcao.replace('R$ ', '').trim();
            const dataUltimoMovimento = detalhes["Data do Último Movimento"] || 'N/A';
            const natureza = detalhes.Assunto || detalhes.Classe || 'N/A';
            scriptTextResponse += `
PROCESSO: ${detalhes["Número do Processo"]}
Requerente (Polo Ativo):
👤 Nome: ${ativoNome}
⚖ Advogado: ${ativoAdvogado}
Requerido (Polo Passivo):
${passivoIcon} Nome: ${passivoNome}
💳 Doc.: ${passivoDocType}: ${passivoDoc || 'N/A'}
Dados da Ação:
⚖ Natureza: ${natureza}
💰 Valor da Causa: ${valorAcao}
🗓 Data do Último Movimento: ${dataUltimoMovimento}
Boa tarde, ${ativoNome}. Espero que esteja bem.
Estou entrando em contato para informar que temos novidades relacionadas ao processo ${detalhes["Número do Processo"]}.
Venho informar que obtivemos êxito na causa, com o valor de ${valorAcao} (${valorPorExtenso(valorCausa)}) liberado em relação ao seu processo.
-----------------------------------------
`;
        }
        const detailsFilePath = path.join(__dirname, `oab_${formattedOab}_detalhes.txt`);
        fs.writeFileSync(detailsFilePath, detailsTextResponse.trim(), 'utf8');
        const detailsMedia = MessageMedia.fromFilePath(detailsFilePath);
        await msg.reply(detailsMedia, undefined, { caption: `✅ Arquivo de texto detalhado para OAB: ${formattedOab}` });
        fs.unlinkSync(detailsFilePath);
        const scriptFilePath = path.join(__dirname, `oab_${formattedOab}_script.txt`);
        fs.writeFileSync(scriptFilePath, scriptTextResponse.trim(), 'utf8');
        const scriptMedia = MessageMedia.fromFilePath(scriptFilePath);
        await msg.reply(scriptMedia, undefined, { caption: `✅ Arquivo de script formatado para OAB: ${formattedOab}` });
        fs.unlinkSync(scriptFilePath);
        logToClient(clientId, `Arquivos para OAB ${formattedOab} enviados com sucesso.`);
    } catch (e) {
        console.error("Erro na consulta OAB:", e);
        await msg.reply("Ocorreu um erro ao realizar a consulta OAB. Verifique o servidor da API.");
        logToClient(clientId, `Erro crítico na consulta OAB: ${e.message}`);
    }
}
async function handleOabConsultaByName(msg, nomeCompleto, client, clientId) {
    try {
        const url = `http://45.158.8.155/consulta_oab.php`;
        const params = { nome: nomeCompleto };
        logToClient(clientId, `Enviando requisição OAB para URL: ${url} com params: ${JSON.stringify(params)}`);
        const response = await axios.get(url, { params });
        const data = response.data;
        if (!data.dados || data.dados.length === 0) {
            await msg.reply('❌ Ops! A consulta não retornou resultados válidos.');
            return;
        }
        // Create a map for UF from first_response
        const ufMap = new Map();
        if (data.first_response && data.first_response.Data) {
            data.first_response.Data.forEach(item => {
                ufMap.set(item.Inscricao, item.UF);
            });
        }
        const results = data.dados;
        for (const result of results) {
            const nome = result.Nome;
            const oab = result.Inscricao;
            const seccional = ufMap.get(oab) || '';
            const formattedOab = `${seccional}${oab}`;
            const message = `*Nome*: ${nome}\n*OAB*: ${formattedOab}`;
            const base64Data = result.base64_image;
            let processedBase64 = base64Data;
            let mimeType = 'image/jpeg';
            if (base64Data && base64Data.startsWith('data:image')) {
                const parts = base64Data.split(',');
                if (parts.length > 1) {
                    mimeType = parts[0].replace('data:', '').replace(';base64', '');
                    processedBase64 = parts[1];
                }
            } else if (base64Data) {
                // Assume raw base64 without data: prefix
                if (!base64Data.startsWith('data:')) {
                    processedBase64 = base64Data; // raw for validation
                }
            }
            if (processedBase64 && /^[A-Za-z0-9+/=]+$/.test(processedBase64)) {
                try {
                    const fullBase64 = `data:${mimeType};base64,${processedBase64}`;
                    const media = new MessageMedia(mimeType, processedBase64, `${nome.replace(/[^a-zA-Z0-9_ ]/g, '')}_oab_${formattedOab}.jpg`);
                    await msg.reply(media, undefined, { caption: message });
                    logToClient(clientId, `Consulta por nome "${nomeCompleto}" realizada com sucesso. Imagem enviada para ${nome}.`);
                } catch (e) {
                    console.error(`[${clientId}] Erro ao enviar imagem OAB para ${nome}:`, e.message);
                    await msg.reply(`${message}\n\n❌ Ops! Houve um problema ao processar a foto.`);
                    logToClient(clientId, `Erro ao enviar imagem para ${nome}: ${e.message}`);
                }
            } else {
                await msg.reply(message);
                logToClient(clientId, `Consulta por nome "${nomeCompleto}" realizada com sucesso. Nenhuma imagem válida encontrada para ${nome}.`);
            }
        }
    } catch (e) {
        console.error(`[${clientId}] Erro ao consultar OAB para ${nomeCompleto}:`, e.message, e.stack);
        await msg.reply('❌ Ops! Ocorreu um erro ao consultar os dados do advogado. Por favor, tente novamente mais tarde.');
        logToClient(clientId, `Erro na consulta por nome: ${e.message}`);
    }
}
async function handleProcessoConsulta(msg, numeroProcesso, client, clientId) {
    try {
        logToClient(clientId, `Realizando consulta na API para o processo: ${numeroProcesso}`);
        const apiResponse = await fetchProcessDetails(numeroProcesso, clientId);
        if (!apiResponse) {
            await msg.reply(`❌ Falha ao consultar o processo ou dados não encontrados.`);
            logToClient(clientId, `Falha na consulta para o processo ${numeroProcesso}.`);
            return;
        }
        const detalhes = apiResponse["Detalhes do Processo"];
        const poloAtivo = apiResponse["Polo Ativo"];
        const poloPassivo = apiResponse["Polo Passivo"];
        const valorAcao = detalhes["Valor da Ação"] || 'N/A';
        const ativoParteStr = poloAtivo?.Parte || '';
        const ativoCpf = extractDocFromParte(ativoParteStr);
        const passivoParteStr = poloPassivo?.Parte || '';
        const passivoDoc = extractDocFromParte(passivoParteStr);
        const ativoCpfData = await getCpfData(ativoCpf);
        const passivoCpfData = passivoDoc && passivoDoc.length === 11 ? await getCpfData(passivoDoc) : null;
        const ativoExtra = ativoCpfData ? `\n${formatCpfData(ativoCpfData)}` : '';
        const passivoExtra = passivoCpfData ? `\n${formatCpfData(passivoCpfData)}` : '';
        const textResponse = `
*Detalhes do Processo:*
-----------------------------------------
*Número do Processo*: ${detalhes["Número do Processo"] || 'N/A'}
*Tribunal*: ${detalhes.Tribunal || 'N/A'}
*Órgão Julgador*: ${detalhes["Órgão Julgador"] || 'N/A'}
*Classe*: ${detalhes.Classe || 'N/A'}
*Assunto*: ${detalhes.Assunto || 'N/A'}
*Valor da Ação*: ${valorAcao}
*Data do Último Movimento*: ${detalhes["Data do Último Movimento"] || 'N/A'}
*Polo Ativo (Autor/Requerente):*
${ativoParteStr}${ativoExtra}
${poloAtivo?.Advogado ? `Advogado: ${poloAtivo.Advogado}` : ''}
*Polo Passivo (Réu/Requerido):*
${passivoParteStr}${passivoExtra}
${poloPassivo?.Advogado ? `Advogado: ${poloPassivo.Advogado}` : ''}
-----------------------------------------
`;
        await msg.reply(textResponse.trim());
        logToClient(clientId, `Detalhes do processo ${numeroProcesso} enviados com sucesso.`);
    } catch (e) {
        console.error(`[${clientId}] Erro na consulta de processo para ${numeroProcesso}:`, e.message, e.stack);
        await msg.reply("❌ Ocorreu um erro ao realizar a consulta do processo. Verifique o servidor da API.");
        logToClient(clientId, `Erro crítico na consulta de processo: ${e.message}`);
    }
}
async function handlePdfGenerationFlowStart(msg, numeroProcesso, client, clientId) {
    try {
        logToClient(clientId, `Comando /pdf ativado para gerar documento do processo ${numeroProcesso}`);
        const apiResponse = await fetchProcessDetails(numeroProcesso, clientId);
        if (!apiResponse) {
            await msg.reply(`❌ Falha ao consultar o processo. A API não retornou os dados necessários.`);
            logToClient(clientId, `Falha na consulta para o processo ${numeroProcesso}.`);
            return;
        }
        const detalhes = apiResponse["Detalhes do Processo"];
        const poloAtivo = apiResponse["Polo Ativo"];
        const poloPassivo = apiResponse["Polo Passivo"];
        let advogadoNome = "N/A";
        if (poloAtivo && poloAtivo.Advogado) {
            const advogadoMatch = poloAtivo.Advogado.match(/Advogado:\s*(.+?)\s*(?:\(CPF:|$)/);
            if (advogadoMatch && advogadoMatch[1]) {
                advogadoNome = advogadoMatch[1].trim();
            }
        }
        const ativoParteStr = poloAtivo?.Parte || '';
        const credorCpf = extractDocFromParte(ativoParteStr) || "N/A";
        const credorNome = ativoParteStr.split('(')[0]?.trim() || "N/A";
        const valor = detalhes["Valor da Ação"]?.replace('R$ ', '').trim() || 'N/A';
        const passivoParteStr = poloPassivo?.Parte || '';
        const parteContraria = passivoParteStr.split('(')[0]?.trim() || "N/A";
        const flowData = {
            parte1_nome: credorNome,
            parte1_cpf: credorCpf,
            parte1_advogado: advogadoNome,
            processo: detalhes["Número do Processo"],
            parte2_nome: parteContraria,
            valor_a_receber: valor
        };
        gerarFlowData.set(msg.from, flowData);
        if (credorNome !== "N/A" &&
            credorCpf !== "N/A" &&
            advogadoNome !== "N/A" &&
            parteContraria !== "N/A") {
            gerarFlowState.set(msg.from, 'confirm_data');
            await sendSummaryMessage(msg.from, flowData, client);
        } else {
            gerarFlowState.set(msg.from, 'ask_missing_info');
            await msg.reply('Algumas informações não puderam ser obtidas automaticamente. Por favor, digite o nome completo do advogado(a):');
        }
    } catch (e) {
        console.error(`[${clientId}] Erro ao iniciar o fluxo de PDF para o processo ${numeroProcesso}:`, e.message, e.stack);
        await msg.reply('❌ Ops! Ocorreu um erro ao consultar os dados para o documento. Por favor, tente novamente mais tarde.');
    }
}
async function handleOabPdfGeneration(msg, formattedOab, numPdfs, client, clientId, oabData) {
    await msg.reply(`✅ Iniciando a consulta de processos para a OAB *${formattedOab}* e geração de PDFs.`);
    logToClient(clientId, `Comando /oabpdf ativado para a OAB ${formattedOab}`);
    try {
        if (!oabData) {
            oabData = await fetchOabProcesses(formattedOab, clientId);
        }
        const processes = oabData.Processos || [];
        const totalProcesses = processes.length;
        if (totalProcesses === 0) {
            await msg.reply(`⚠️ Nenhum processo encontrado para a OAB: *${formattedOab}*.`);
            return;
        }
        if (numPdfs === null) {
            oabFlowState.set(msg.from, 'ask_num_pdfs');
            oabFlowData.set(msg.from, { maxPdfs: Math.min(totalProcesses, MAX_PDFS), formattedOab, command: 'oabpdf', oabData });
            await msg.reply(`Encontrados ${totalProcesses} processos. Quantos PDFs deseja gerar (1 a ${oabFlowData.get(msg.from).maxPdfs})?`);
            return;
        }
        if (numPdfs > MAX_PDFS) {
            await msg.reply(`O máximo permitido é ${MAX_PDFS} PDFs.`);
            return;
        }
        await msg.reply(`⌛ Encontrados *${totalProcesses}* processos. Processando...`);
        processes.sort((a, b) => {
            const dateStrA = a["Detalhes do Processo"]["Data do Último Movimento"] || '01/01/1970 00:00';
            const dateStrB = b["Detalhes do Processo"]["Data do Último Movimento"] || '01/01/1970 00:00';
            const [datePartA, timePartA] = dateStrA.split(' ');
            const [ddA, mmA, yyyyA] = datePartA.split('/');
            const [hhA, minA] = timePartA.split(':');
            const dateA = new Date(yyyyA, mmA - 1, ddA, hhA, minA);
            const [datePartB, timePartB] = dateStrB.split(' ');
            const [ddB, mmB, yyyyB] = datePartB.split('/');
            const [hhB, minB] = timePartB.split(':');
            const dateB = new Date(yyyyB, mmB - 1, ddB, hhB, minB);
            if (dateA.getTime() !== dateB.getTime()) {
                return dateB.getTime() - dateA.getTime();
            }
            const valorA = parseFloat(a["Detalhes do Processo"]["Valor da Ação"]?.replace('R$ ', '').replace(/\./g, '').replace(',', '.') || 0);
            const valorB = parseFloat(b["Detalhes do Processo"]["Valor da Ação"]?.replace('R$ ', '').replace(/\./g, '').replace(',', '.') || 0);
            return valorB - valorA;
        });
        const processesToGenerate = processes.slice(0, numPdfs);
        if (processesToGenerate.length === 0) {
            await msg.reply('❌ Nenhum dado foi encontrado para gerar os PDFs.');
            return;
        }
        await msg.reply(`✅ Coletado dados para *${processesToGenerate.length}* processos. Agora, iniciando a geração e envio dos PDFs.`);
        const batchSize = numPdfs > 100 ? 50 : processesToGenerate.length;
        let batchNum = 1;
        const numBatches = Math.ceil(processesToGenerate.length / batchSize);
        for (let i = 0; i < processesToGenerate.length; i += batchSize) {
            const batch = processesToGenerate.slice(i, i + batchSize);
            await msg.reply(`Processando lote ${batchNum} de ${numBatches}...`);
            const pdfGenerationPromises = batch.map(async (process) => {
                const detalhes = process["Detalhes do Processo"];
                const poloAtivo = process["Polo Ativo"];
                const poloPassivo = process["Polo Passivo"] || {};
                if (!detalhes || !poloAtivo) {
                    return Promise.reject(new Error(`Dados incompletos para o processo ${detalhes["Número do Processo"]}`));
                }
                const advogadoNome = poloAtivo.Advogado?.match(/Advogado:\s*(.+?)\s*(?:\(CPF:|$)/)?.[1]?.trim() || "N/A";
                const ativoParteStr = poloAtivo.Parte || '';
                const credorCpf = extractDocFromParte(ativoParteStr) || "N/A";
                const credorNome = ativoParteStr.split('(')[0]?.trim() || "N/A";
                const passivoParteStr = poloPassivo.Parte || '';
                const parteContraria = passivoParteStr.split('(')[0]?.trim() || "N/A";
                const valor = detalhes["Valor da Ação"]?.replace('R$ ', '').trim() || 'N/A';
                const pdfData = {
                    parte1_nome: credorNome,
                    parte1_cpf: credorCpf,
                    parte1_advogado: advogadoNome,
                    processo: detalhes["Número do Processo"],
                    parte2_nome: parteContraria,
                    valor_a_receber: valor
                };
                return generateAndSendPdf(msg.from, pdfData, client, clientId);
            });
            const results = await Promise.allSettled(pdfGenerationPromises);
            const failedGenerations = results.filter(result => result.status === 'rejected').length;
            if (failedGenerations > 0) {
                await msg.reply(`⚠️ Aviso: ${failedGenerations} PDF(s) no lote ${batchNum} não puderam ser gerados devido a erros.`);
            }
            await new Promise(resolve => setTimeout(resolve, 2000)); // Delay entre lotes
            batchNum++;
        }
        await msg.reply('✅ Atividade finalizada! Todos os PDFs gerados foram enviados.');
    } catch (e) {
        console.error(`[${clientId}] Erro geral no fluxo de /oabpdf:`, e.message);
        await msg.reply('❌ Ocorreu um erro inesperado. Por favor, tente novamente mais tarde.');
    }
}
async function handleOabZipGeneration(msg, formattedOab, numPdfs, client, clientId, oabData) {
    try {
        if (!sessions.get(clientId)?.connected) {
            await msg.reply('⚠️ Sessão desconectada. Por favor, reconecte e tente novamente.');
            logToClient(clientId, 'Erro: Sessão desconectada durante /oabzip.');
            return;
        }
        if (!sessions.has(clientId) || !sessions.get(clientId).client) {
            await msg.reply('⚠️ Sessão inválida ou cliente não inicializado. Por favor, reinicie a sessão.');
            logToClient(clientId, 'Erro: Sessão inválida durante /oabzip.');
            return;
        }
        await msg.reply(`✅ Iniciando a consulta de processos para a OAB *${formattedOab}* e geração de ZIP com PDFs.`);
        logToClient(clientId, `Comando /oabzip ativado para a OAB ${formattedOab}`);
        const tempDir = path.join(__dirname, 'tempzips');
        if (!fs.existsSync(tempDir)) {
            fs.mkdirSync(tempDir, { recursive: true });
        }
        if (!oabData) {
            oabData = await fetchOabProcesses(formattedOab, clientId);
        }
        const processes = oabData.Processos || [];
        const totalProcesses = processes.length;
        if (totalProcesses === 0) {
            await msg.reply(`⚠️ Nenhum processo encontrado para a OAB: *${formattedOab}*.`);
            return;
        }
        if (numPdfs === null) {
            oabFlowState.set(msg.from, 'ask_num_pdfs');
            oabFlowData.set(msg.from, { maxPdfs: Math.min(totalProcesses, MAX_PDFS), formattedOab, command: 'oabzip', oabData });
            await msg.reply(`Encontrados ${totalProcesses} processos. Quantos PDFs deseja gerar (1 a ${oabFlowData.get(msg.from).maxPdfs})?`);
            return;
        }
        if (numPdfs > MAX_PDFS) {
            await msg.reply(`O máximo permitido é ${MAX_PDFS} PDFs.`);
            return;
        }
        await msg.reply(`⌛ Encontrados *${totalProcesses}* processos. Processando...`);
        processes.sort((a, b) => {
            const dateStrA = a["Detalhes do Processo"]["Data do Último Movimento"] || '01/01/1970 00:00';
            const dateStrB = b["Detalhes do Processo"]["Data do Último Movimento"] || '01/01/1970 00:00';
            const [datePartA, timePartA] = dateStrA.split(' ');
            const [ddA, mmA, yyyyA] = datePartA.split('/');
            const [hhA, minA] = timePartA.split(':');
            const dateA = new Date(yyyyA, mmA - 1, ddA, hhA, minA);
            const [datePartB, timePartB] = dateStrB.split(' ');
            const [ddB, mmB, yyyyB] = datePartB.split('/');
            const [hhB, minB] = timePartB.split(':');
            const dateB = new Date(yyyyB, mmB - 1, ddB, hhB, minB);
            if (dateA.getTime() !== dateB.getTime()) {
                return dateB.getTime() - dateA.getTime();
            }
            const valorA = parseFloat(a["Detalhes do Processo"]["Valor da Ação"]?.replace('R$ ', '').replace(/\./g, '').replace(',', '.') || 0);
            const valorB = parseFloat(b["Detalhes do Processo"]["Valor da Ação"]?.replace('R$ ', '').replace(/\./g, '').replace(',', '.') || 0);
            return valorB - valorA;
        });
        const processesToGenerate = processes.slice(0, numPdfs);
        if (processesToGenerate.length === 0) {
            await msg.reply('❌ Nenhum dado foi encontrado para gerar o ZIP.');
            return;
        }
        await msg.reply(`✅ Coletado dados para *${processesToGenerate.length}* processos. Gerando ZIPs em lotes de 20...`);
        const batchSize = 20;
        let batchNum = 1;
        const numBatches = Math.ceil(processesToGenerate.length / batchSize);
        for (let i = 0; i < processesToGenerate.length; i += batchSize) {
            if (!sessions.get(clientId)?.connected) {
                await msg.reply('⚠️ Sessão foi desconectada durante o processamento. Operação abortada.');
                logToClient(clientId, 'Sessão desconectada durante processamento de lote.');
                return;
            }
            const batch = processesToGenerate.slice(i, i + batchSize);
            const pdfFiles = [];
            let detailsTextResponse = `*Consulta Detalhada OAB: ${formattedOab} - Lote ${batchNum}*\n\n`;
            let scriptTextResponse = `*Script OAB: ${formattedOab} - Lote ${batchNum}*\n\n`;
            for (const process of batch) {
                if (!sessions.get(clientId)?.connected) {
                    await msg.reply('⚠️ Sessão foi desconectada durante o processamento de um processo. Operação abortada.');
                    logToClient(clientId, 'Sessão desconectada durante processamento de processo individual.');
                    return;
                }
                try {
                    const detalhes = process["Detalhes do Processo"];
                    const poloAtivo = process["Polo Ativo"];
                    const poloPassivo = process["Polo Passivo"] || {};
                    if (!detalhes || !poloAtivo) {
                        logToClient(clientId, `Dados incompletos para o processo ${detalhes["Número do Processo"] || 'desconhecido'}. Pulando.`);
                        continue;
                    }
                    const advogadoNome = poloAtivo.Advogado?.match(/Advogado:\s*(.+?)\s*(?:\(CPF:|$)/)?.[1]?.trim() || "N/A";
                    const ativoParteStr = poloAtivo.Parte || '';
                    const credorCpf = extractDocFromParte(ativoParteStr) || "N/A";
                    const credorNome = ativoParteStr.split('(')[0]?.trim() || "N/A";
                    const passivoParteStr = poloPassivo.Parte || '';
                    const parteContraria = passivoParteStr.split('(')[0]?.trim() || "N/A";
                    const valor = detalhes["Valor da Ação"]?.replace('R$ ', '').trim() || 'N/A';
                    const pdfData = {
                        parte1_nome: credorNome,
                        parte1_cpf: credorCpf,
                        parte1_advogado: advogadoNome,
                        processo: detalhes["Número do Processo"],
                        parte2_nome: parteContraria,
                        valor_a_receber: valor
                    };
                    const url = `http://45.158.8.155:7261/gerar_oficio?credor=${encodeURIComponent(pdfData.parte1_nome)}&cpf=${encodeURIComponent(pdfData.parte1_cpf)}&advogado=${encodeURIComponent(pdfData.parte1_advogado)}&processo=${encodeURIComponent(pdfData.processo)}&parte_contraria=${encodeURIComponent(pdfData.parte2_nome)}&valor=${encodeURIComponent(pdfData.valor_a_receber)}`;
                    const response = await fetchWithRetry(url, 3);
                    const pdfBuffer = Buffer.from(response.data);
                    const pdfFilename = `Alvara_Liberacao_${pdfData.parte1_nome.replace(/[^a-zA-Z0-9_ ]/g, '')}_${pdfData.processo.replace(/[^a-zA-Z0-9]/g, '')}.pdf`;
                    pdfFiles.push({ buffer: pdfBuffer, name: pdfFilename });
                    logToClient(clientId, `✅ PDF gerado com sucesso para processo ${pdfData.processo}`);
                    const valorAcao = detalhes["Valor da Ação"] || 'N/A';
                    const ativoCpf = credorCpf;
                    const passivoDoc = extractDocFromParte(passivoParteStr);
                    const ativoCpfData = await getCpfData(ativoCpf);
                    const passivoCpfData = passivoDoc && passivoDoc.length === 11 ? await getCpfData(passivoDoc) : null;
                    detailsTextResponse += `
-----------------------------------------
*Processo:* ${detalhes["Número do Processo"] || 'N/A'}
*Tribunal:* ${detalhes.Tribunal || 'N/A'}
*Órgão Julgador*: ${detalhes["Órgão Julgador"] || 'N/A'}
*Classe*: ${detalhes.Classe || 'N/A'}
*Assunto*: ${detalhes.Assunto || 'N/A'}
*Valor da Ação*: ${valorAcao}
*Data do Último Movimento*: ${detalhes["Data do Último Movimento"] || 'N/A'}
*Polo Ativo:* ${ativoParteStr}${ativoCpfData ? ` (${formatCpfData(ativoCpfData)})` : ''}
*Polo Passivo:* ${passivoParteStr}${passivoCpfData ? ` (${formatCpfData(passivoCpfData)})` : ''}
-----------------------------------------
`;
                    const ativoNome = ativoParteStr.split('(')[0]?.trim() || 'N/A';
                    const ativoAdvogado = poloAtivo.Advogado?.match(/Advogado:\s*(.+?)\s*(?:\(CPF:|$)/)?.[1]?.trim() || 'N/A';
                    const passivoNome = passivoParteStr.split('(')[0]?.trim() || 'N/A';
                    const passivoIcon = passivoDoc && passivoDoc.length === 14 ? '🏢' : '👤';
                    const passivoDocType = passivoDoc && passivoDoc.length === 14 ? 'CNPJ' : 'CPF';
                    const valorCausa = valor;
                    const dataUltimoMovimento = detalhes["Data do Último Movimento"] || 'N/A';
                    const natureza = detalhes.Assunto || detalhes.Classe || 'N/A';
                    scriptTextResponse += `
PROCESSO: ${detalhes["Número do Processo"]}
Requerente (Polo Ativo):
👤 Nome: ${ativoNome}
⚖ Advogado: ${ativoAdvogado}
Requerido (Polo Passivo):
${passivoIcon} Nome: ${passivoNome}
💳 Doc.: ${passivoDocType}: ${passivoDoc || 'N/A'}
Dados da Ação:
⚖ Natureza: ${natureza}
💰 Valor da Causa: R$ ${valorCausa}
🗓 Data do Último Movimento: ${dataUltimoMovimento}
Boa tarde, ${ativoNome}. Espero que esteja bem.
Estou entrando em contato para informar que temos novidades relacionadas ao processo ${detalhes["Número do Processo"]}.
Venho informar que obtivemos êxito na causa, com o valor de R$ ${valorCausa} (${valorPorExtenso(valorCausa)}) liberado em relação ao seu processo.
-----------------------------------------
`;
                } catch (pdfErr) {
                    console.error(`[${clientId}] Erro ao gerar PDF para o processo ${process["Detalhes do Processo"]["Número do Processo"] || 'desconhecido'} em lote ${batchNum}:`, pdfErr.message, pdfErr.stack);
                    logToClient(clientId, `❌ Erro ao gerar PDF para processo ${process["Detalhes do Processo"]["Número do Processo"] || 'desconhecido'} em lote ${batchNum}: ${pdfErr.message}`);
                }
                // Pequeno delay entre processos para evitar sobrecarga
                await new Promise(resolve => setTimeout(resolve, 500));
            }
            const zipFilePath = path.join(tempDir, `oab_${formattedOab}_arquivos_lote_${batchNum}.zip`);
            const output = fs.createWriteStream(zipFilePath);
            const archive = archiver('zip', { zlib: { level: 9 } });
            archive.pipe(output);
            for (const file of pdfFiles) {
                archive.append(file.buffer, { name: file.name });
                // Limpa o buffer imediatamente após append para liberar memória
                file.buffer = null;
            }
            const detailsFileName = `detalhes_lote_${batchNum}.txt`;
            archive.append(detailsTextResponse, { name: detailsFileName });
            const scriptFileName = `script_lote_${batchNum}.txt`;
            archive.append(scriptTextResponse, { name: scriptFileName });
            await archive.finalize();
            await new Promise((resolve, reject) => {
                output.on('close', resolve);
                archive.on('error', reject);
            });
            const zipMedia = MessageMedia.fromFilePath(zipFilePath);
            const caption = `✅ ZIP com PDFs, detalhes e script para OAB: ${formattedOab} - Lote ${batchNum} de ${numBatches}`;
            await msg.reply(zipMedia, undefined, { caption });
            fs.unlinkSync(zipFilePath);
            logToClient(clientId, `ZIP para OAB ${formattedOab} lote ${batchNum} enviado com sucesso.`);
            batchNum++;
            // Delay maior entre lotes para evitar sobrecarga de rede/sessão (aumentado para 5s em grandes quantidades)
            const batchDelay = numPdfs > 500 ? 5000 : 3000;
            await new Promise(resolve => setTimeout(resolve, batchDelay));
        }
    } catch (e) {
        console.error(`[${clientId}] Erro geral no fluxo de /oabzip (possivelmente Target closed):`, e.message, e.stack);
        await msg.reply('❌ Ocorreu um erro inesperado (possivelmente sessão fechada). Por favor, reconecte a sessão e tente novamente mais tarde.');
    } finally {
        const tempDir = path.join(__dirname, 'tempzips');
        if (fs.existsSync(tempDir)) {
            fs.rmSync(tempDir, { recursive: true, force: true });
            logToClient(clientId, 'Diretório temporário limpo com sucesso.');
        }
    }
}
async function generateAndSendPdf(chatId, pdfData, client, clientId) {
    const startTime = Date.now();
    try {
        const url = `http://45.158.8.155:7261/gerar_oficio?credor=${encodeURIComponent(pdfData.parte1_nome)}&cpf=${encodeURIComponent(pdfData.parte1_cpf)}&advogado=${encodeURIComponent(pdfData.parte1_advogado)}&processo=${encodeURIComponent(pdfData.processo)}&parte_contraria=${encodeURIComponent(pdfData.parte2_nome)}&valor=${encodeURIComponent(pdfData.valor_a_receber)}`;
        logToClient(clientId, `🔄 Gerando PDF para processo ${pdfData.processo}...`);
        console.log(`[${clientId}] Iniciando geração PDF - Processo: ${pdfData.processo}, Credor: ${pdfData.parte1_nome}`);
        const response = await fetchWithRetry(url, 3);
        const pdfBuffer = Buffer.from(response.data);
        const generationTime = (Date.now() - startTime) / 1000;
        console.log(`[${clientId}] PDF gerado em ${generationTime}s - Tamanho: ${pdfBuffer.length} bytes`);
        const [detalhesProcesso] = await Promise.all([
            fetchProcessDetails(pdfData.processo, clientId)
        ]);
        let detalhesTexto = "";
        if (detalhesProcesso) {
            const detalhes = detalhesProcesso["Detalhes do Processo"];
            const poloAtivo = detalhesProcesso["Polo Ativo"];
            const poloPassivo = detalhesProcesso["Polo Passivo"];
            const valorAcao = detalhes["Valor da Ação"] || 'N/A';
            detalhesTexto = `
*📋 DETALHES COMPLETOS DO PROCESSO*
-----------------------------------------
*Número do Processo*: ${detalhes["Número do Processo"] || 'N/A'}
*Tribunal*: ${detalhes.Tribunal || 'N/A'}
*Órgão Julgador*: ${detalhes["Órgão Julgador"] || 'N/A'}
*Classe*: ${detalhes.Classe || 'N/A'}
*Assunto*: ${detalhes.Assunto || 'N/A'}
*Valor da Ação*: ${valorAcao}
*Data do Último Movimento*: ${detalhes["Data do Último Movimento"] || 'N/A'}
*👤 POLO ATIVO (Autor/Requerente):*
${poloAtivo?.Parte || 'N/A'}
${poloAtivo?.Advogado ? `Advogado: ${poloAtivo.Advogado}` : ''}
*⚖️ POLO PASSIVO (Réu/Requerido):*
${poloPassivo?.Parte || 'N/A'}
${poloPassivo?.Advogado ? `Advogado: ${poloPassivo.Advogado}` : ''}
-----------------------------------------
*⏱️ Gerado em ${generationTime.toFixed(1)}s*
            `;
        }
        const filename = `Alvara_Liberacao_${pdfData.parte1_nome.replace(/[^a-zA-Z0-9_ ]/g, '')}_${pdfData.processo.replace(/[^a-zA-Z0-9]/g, '')}.pdf`;
        const media = new MessageMedia('application/pdf', pdfBuffer.toString('base64'), filename);
        const detailedCaption = `✅ *ALVARÁ DE LIBERAÇÃO DE PAGAMENTO GERADO!*
*📄 DOCUMENTO PRONTO PARA DOWNLOAD*
*Resumo da Solicitação:*
• *👤 Credor*: ${pdfData.parte1_nome}
• *🆔 CPF*: ${pdfData.parte1_cpf}
• *⚖️ Advogado*: ${pdfData.parte1_advogado}
• *📋 Processo*: ${pdfData.processo}
• *👥 Parte Contrária*: ${pdfData.parte2_nome}
• *💰 Valor*: R$ ${pdfData.valor_a_receber}
${detalhesTexto}
*📱 Use este documento para liberação judicial dos valores.
*⚠️ Mantenha o arquivo seguro e apresente apenas quando solicitado pelo cartório/banco.*`;
        await client.sendMessage(chatId, media, { caption: detailedCaption });
        const totalTime = (Date.now() - startTime) / 1000;
        logToClient(clientId, `✅ PDF ENVIADO com sucesso para ${pdfData.processo} em ${totalTime.toFixed(1)}s`);
        console.log(`[${clientId}] PDF enviado com sucesso - Total: ${totalTime}s`);
        if (gerarFlowState.has(chatId)) {
            gerarFlowState.delete(chatId);
            gerarFlowData.delete(chatId);
            await client.sendMessage(chatId, '🎉 *PROCESSO CONCLUÍDO COM SUCESSO!*\n\nObrigado por usar nosso serviço. Para nova solicitação, use */menu* ou */gerar*.');
        }
        // Delay após envio para grandes quantidades no /oabpdf
        if (process.env.LARGE_PDF_DELAY) {
            await new Promise(resolve => setTimeout(resolve, 1000));
        }
    } catch (e) {
        const totalTime = (Date.now() - startTime) / 1000;
        console.error(`[${clientId}] ❌ ERRO PDF para ${chatId} após ${totalTime}s:`, e.message);
        const errorMessage = `❌ *ERRO NA GERAÇÃO DO DOCUMENTO*
*O que aconteceu:*
• Falha na comunicação com o servidor de documentos
• Tempo total da operação: ${totalTime.toFixed(1)}s
*Dados que foram processados:*
• *Processo*: ${pdfData.processo}
• *Credor*: ${pdfData.parte1_nome}
• *Valor*: R$ ${pdfData.valor_a_receber}
*🔄 Tente novamente em 1-2 minutos.
*📞 Se o erro persistir, contate o suporte.*
*Motivo técnico*: ${e.message}`;
        await client.sendMessage(chatId, errorMessage);
        logToClient(clientId, `❌ FALHA PDF ${pdfData.processo}: ${e.message}`);
        if (gerarFlowState.has(chatId)) {
            gerarFlowState.delete(chatId);
            gerarFlowData.delete(chatId);
        }
    }
}
async function sendSummaryMessage(chatId, flowData, client, isCorrection = false) {
    const summaryMessage = [
        isCorrection ? 'Campo corrigido. Por favor, confirme os dados novamente:' :
            'Dados obtidos automaticamente do processo. Por favor, confirme:',
        `*1. Nome*: ${flowData.parte1_nome}`,
        `*2. CPF*: ${flowData.parte1_cpf}`,
        `*3. Advogado*: ${flowData.parte1_advogado}`,
        `*4. Processo*: ${flowData.processo}`,
        `*5. Parte Contrária*: ${flowData.parte2_nome}`,
        `*6. Valor*: ${flowData.valor_a_receber}`,
        '',
        'As informações estão corretas?',
        '✅ *Sim* (para confirmar e gerar PDF) ou ❌ *Não* (para corrigir).'
    ].join('\n');
    await client.sendMessage(chatId, summaryMessage);
}
app.use(express.json());
app.post('/criar-sessao', (req, res) => {
    const { clientId } = req.body;
    if (!clientId || !isValidClientId(clientId)) {
        return res.status(400).json({ success: false, message: "ID da sessão é inválido. Apenas caracteres alfanuméricos, underscores e hyphens são permitidos." });
    }
    if (sessions.has(clientId)) {
        return res.status(400).json({ success: false, message: "ID da sessão já existe." });
    }
    if (sessions.size >= MAX_SESSIONS) {
        return res.status(400).json({ success: false, message: `Limite de ${MAX_SESSIONS} sessões atingido.` });
    }
    sessions.set(clientId, { clientId, status: "Nova", connected: false, client: null });
    broadcastSessionsUpdate();
    saveSessions();
    res.json({ success: true });
});
app.post('/iniciar-sessao', (req, res) => {
    initializeSession(req.body.clientId);
    res.json({ success: true, message: "Sessão iniciada ou já em andamento." });
});
app.post('/initialize-all-sequentially', async (req, res) => {
    const sessionIds = Array.from(sessions.keys());
    if (sessionIds.length === 0) {
        return res.json({ success: true, message: "Nenhuma sessão encontrada para inicializar." });
    }
    console.log("Iniciando sessões sequencialmente:", sessionIds);
    res.json({ success: true, message: `Iniciando ${sessionIds.length} sessões sequencialmente. Verifique os logs.` });
    (async () => {
        try {
            for (const clientId of sessionIds) {
                console.log(`Iniciando a sessão ${clientId}...`);
                await initializeSession(clientId);
                console.log(`Sessão ${clientId} inicializada. Prosseguindo para a próxima.`);
            }
            console.log("Processo de inicialização sequencial finalizado.");
        } catch (error) {
            console.error("Erro na inicialização sequencial:", error);
        }
    })();
});
app.post('/desconectar', (req, res) => {
    destroySession(req.body.clientId, req.body.removeFromDisk);
    res.json({ success: true });
});
app.get('/sessions', (req, res) => {
    const sessionStates = {};
    sessions.forEach((s, id) => {
        sessionStates[id] = { clientId: id, status: s.status, qrCodeUrl: s.qrCodeUrl, connected: s.connected };
    });
    res.json(sessionStates);
});
app.get('/oab/:oab', async (req, res) => {
    const oab = req.params.oab;
    try {
        const response = await axios.get(`http://127.0.0.1:5000/consulta?oab=${oab}`);
        if (response.status !== 200) {
            throw new Error(`Erro na consulta: ${response.statusText}`);
        }
        const data = response.data;
        res.json({ "Consulta por OAB": data });
    } catch (e) {
        res.status(500).json({ error: "Falha na consulta OAB", details: e.message });
    }
});
app.get('/processo/:processo', async (req, res) => {
    const numeroProcesso = req.params.processo;
    try {
        const response = await axios.get(`http://127.0.0.1:5000/consulta?processo=${numeroProcesso}`);
        if (response.status !== 200) {
            throw new Error(`Erro na consulta: ${response.statusText}`);
        }
        const data = response.data;
        res.json({ "Consulta por Processo": data });
    } catch (e) {
        res.status(500).json({ error: "Falha na consulta de processo", details: e.message });
    }
});
app.get('/interface', (req, res) => {
    res.send(`<!DOCTYPE html>
    <html>
    <head>
      <meta charset="UTF-8">
      <title>Painel de Sessões WhatsApp</title>
      <script src="/socket.io/socket.io.js"></script>
      <style>
        :root {
          --bg-color: #1a1a1a;
          --card-bg: #2a2a2a;
          --text-color: #f0f0f0;
          --accent-color: #25d366;
          --danger-color: #dc3545;
          --border-color: #444;
          --shadow-light: rgba(0, 0, 0, 0.2);
          --shadow-dark: rgba(0, 0, 0, 0.4);
        }
        body {
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, Arial, sans-serif;
          background-color: var(--bg-color);
          color: var(--text-color);
          margin: 0;
          padding: 2rem;
          display: flex;
          flex-direction: column;
          align-items: center;
          gap: 2rem;
        }
        .panel {
          width: 100%;
          max-width: 900px;
          background-color: var(--card-bg);
          padding: 2rem;
          border-radius: 12px;
          box-shadow: 0 8px 16px var(--shadow-dark), 0 4px 8px var(--shadow-light);
          transition: transform 0.3s ease;
        }
        .panel:hover {
          transform: translateY(-5px);
        }
        h2 {
          margin-top: 0;
          border-bottom: 2px solid var(--accent-color);
          padding-bottom: 0.5rem;
          margin-bottom: 1.5rem;
        }
        button {
          padding: 0.75rem 1.5rem;
          border: none;
          border-radius: 8px;
          cursor: pointer;
          font-weight: bold;
          transition: background-color 0.3s ease, transform 0.1s ease;
        }
        button:hover {
          transform: translateY(-2px);
        }
        .btn-connect {
          background-color: var(--accent-color);
          color: var(--card-bg);
        }
        .btn-connect:hover {
          background-color: #128c7e;
        }
        .btn-danger {
          background-color: var(--danger-color);
          color: var(--text-color);
        }
        .btn-danger:hover {
          background-color: #a71d2a;
        }
        .btn-secondary {
          background-color: var(--border-color);
          color: var(--text-color);
        }
        .btn-secondary:hover {
          background-color: #666;
        }
        input[type="text"] {
          padding: 0.75rem;
          border-radius: 8px;
          border: 1px solid var(--border-color);
          background-color: #333;
          color: var(--text-color);
          width: 100%;
          box-sizing: border-box;
        }
        .action-row {
          display: flex;
          gap: 1rem;
          align-items: center;
          margin-bottom: 1rem;
        }
        .session-card {
          display: grid;
          grid-template-columns: 1fr auto auto 150px;
          align-items: center;
          gap: 1.5rem;
          padding: 1rem;
          border-bottom: 1px solid var(--border-color);
        }
        .session-card:last-child {
          border-bottom: none;
        }
        .session-info {
          display: flex;
          flex-direction: column;
        }
        .session-info strong {
          font-size: 1.1rem;
        }
        .status {
          font-size: 0.9rem;
          color: #999;
        }
        .status.connected {
          color: var(--accent-color);
        }
        .qr-code img {
          display: block;
          border: 2px solid var(--accent-color);
          border-radius: 8px;
          cursor: pointer;
        }
        .no-sessions {
          text-align: center;
          color: #999;
          font-style: italic;
          padding: 2rem;
        }
        .modal {
          display: none;
          position: fixed;
          z-index: 10;
          left: 0;
          top: 0;
          width: 100%;
          height: 100%;
          overflow: auto;
          background-color: rgba(0, 0, 0, 0.8);
          justify-content: center;
          align-items: center;
          backdrop-filter: blur(5px);
        }
        .modal-content {
          background-color: var(--card-bg);
          padding: 2rem;
          border-radius: 12px;
          box-shadow: 0 5px 15px rgba(0,0,0,0.5);
          max-width: 90%;
          text-align: center;
        }
        .modal-content img {
          width: 300px;
          height: 300px;
          border: 4px solid var(--accent-color);
          border-radius: 12px;
        }
        .log-panel {
            max-height: 400px;
            overflow-y: auto;
            font-family: monospace;
            background-color: #111;
            padding: 1rem;
            border-radius: 8px;
            border: 1px solid var(--border-color);
            margin-top: 1.5rem;
        }
        .log-entry {
            margin-bottom: 0.5rem;
        }
        .log-timestamp {
            color: #888;
        }
        .log-session {
            color: var(--accent-color);
            font-weight: bold;
        }
        @media (max-width: 600px) {
          body { padding: 1rem; }
          .panel { padding: 1rem; }
          .session-card {
            grid-template-columns: 1fr;
            text-align: center;
          }
          .action-row {
              flex-direction: column;
          }
          .modal-content img {
              width: 200px;
              height: 200px;
          }
        }
      </style>
    </head>
    <body>
  
    <div class="panel">
      <h2>Sessões Conectadas</h2>
      <div id="connected-sessions"></div>
    </div>
  
    <div class="panel">
      <h2>Gerenciar Sessões</h2>
      <div class="action-row">
        <input type="text" id="newId" placeholder="Digite o ID da nova sessão">
        <button class="btn-connect" onclick="createSession()">Criar Sessão</button>
      </div>
      <div class="action-row">
        <button class="btn-connect" onclick="initializeAllSequentially()">Inicializar todas sequencialmente</button>
      </div>
      <div id="all-sessions"></div>
    </div>
  
    <div class="panel">
      <h2>Logs de Atividade</h2>
      <div id="log-panel" class="log-panel"></div>
    </div>
  
    <div id="qr-modal" class="modal" onclick="closeModal()">
      <div class="modal-content" onclick="event.stopPropagation()">
        <h3>Escaneie o QR Code</h3>
        <img id="modal-qr-img" src="" alt="QR Code">
        <p id="modal-status"></p>
      </div>
    </div>
  
    <script>
    const socket = io();
    const qrModal = document.getElementById('qr-modal');
    const modalQrImg = document.getElementById('modal-qr-img');
    const modalStatus = document.getElementById('modal-status');
    const logPanel = document.getElementById('log-panel');
  
    socket.on('sessions_update', render);
    socket.on('log_update', log => {
      const logEntry = document.createElement('div');
      logEntry.classList.add('log-entry');
      logEntry.innerHTML = \`<span class="log-timestamp">[\${log.timestamp}]</span> <span class="log-session">\${log.clientId}</span>: \${log.message}\`;
      logPanel.prepend(logEntry);
      if (logPanel.children.length > 50) {
        logPanel.removeChild(logPanel.lastChild);
      }
    });
  
    window.onload = function() {
      fetch('/sessions')
        .then(response => response.json())
        .then(sessions => {
          render(sessions);
        })
        .catch(error => console.error('Erro ao carregar sessões:', error));
    };
  
    function render(sessions){
      const connectedPanel = document.getElementById('connected-sessions');
      const allPanel = document.getElementById('all-sessions');
    
      connectedPanel.innerHTML = '';
      allPanel.innerHTML = '';
  
      const currentSessions = Object.values(sessions);
  
      const connectedSessions = currentSessions.filter(s => s.connected);
      if (connectedSessions.length > 0) {
        connectedSessions.forEach(s => {
          connectedPanel.innerHTML += \`<div class="session-card">
              <div class="session-info">
                <strong>\${s.clientId}</strong>
                <span class="status connected">\${s.status}</span>
              </div>
              <div>✅ Conectado</div>
              <div>
                <button class="btn-danger" onclick="disconnect('\${s.clientId}')">Desconectar 🔌</button>
              </div>
            </div>\`;
        });
      } else {
        connectedPanel.innerHTML = '<p class="no-sessions">Nenhuma sessão conectada no momento.</p>';
      }
  
      currentSessions.forEach(s => {
        allPanel.innerHTML += \`<div class="session-card">
          <div class="session-info">
            <strong>\${s.clientId}</strong>
            <span class="status">\${s.status}</span>
          </div>
          <div class="qr-code">\${s.qrCodeUrl ? \`<img src="\${s.qrCodeUrl}" width=100 onclick="showQrModal('\${s.clientId}', '\${s.qrCodeUrl}')">\` : ''}</div>
          <div style="display: flex; gap: 0.5rem; flex-direction: column;">
            \${s.connected ? \`<button class="btn-danger" onclick="disconnect('\${s.clientId}')">Desconectar 🔌</button>\` : \`<button class="btn-connect" onclick="connect('\${s.clientId}')">Conectar ▶️</button>\`}
            <button class="btn-secondary" onclick="deleteSession('\${s.clientId}')">Excluir 🗑️</button>
          </div>
        </div>\`;
      });
    
      const sessionWithQr = currentSessions.find(s => s.qrCodeUrl && s.status === "Aguardando QR Code");
      if (sessionWithQr) {
        showQrModal(sessionWithQr.clientId, sessionWithQr.qrCodeUrl);
      } else {
        closeModal();
      }
    }
  
    function showQrModal(clientId, qrCodeUrl) {
      modalQrImg.src = qrCodeUrl;
      modalStatus.textContent = \`Aguardando conexão da sessão: \${clientId}\`;
      qrModal.style.display = "flex";
    }
  
    function closeModal() {
      qrModal.style.display = "none";
    }
  
    async function createSession(){
      const id=document.getElementById('newId').value;
      const response = await fetch('/criar-sessao',{
        method:'POST',
        headers:{'Content-Type':'application/json'},
        body:JSON.stringify({clientId:id})
      });
      const data = await response.json();
      if (data.success) {
        document.getElementById('newId').value = '';
      } else {
        alert(data.message);
      }
    }
    async function connect(id){await fetch('/iniciar-sessao',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({clientId:id})});}
    async function initializeAllSequentially(){await fetch('/initialize-all-sequentially',{method:'POST',headers:{'Content-Type':'application/json'}});}
    async function disconnect(id){await fetch('/desconectar',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({clientId:id,removeFromDisk:false})});}
    async function deleteSession(id){await fetch('/desconectar',{method:'POST',headers:{'Content-Type':'application/json'},body:JSON.stringify({clientId:id,removeFromDisk:true})});}
    </script>
    </body>
    </html>`);
});
loadSessions();
server.listen(PORT, () => console.log("Rodando em http://localhost:" + PORT + "/interface"));