﻿from flask import Flask, request, jsonify
import json
import cloudscraper
from lxml import html
from selenium.webdriver.chrome.options import Options
from selenium.webdriver.common.by import By
from selenium.webdriver.support.ui import WebDriverWait
from selenium.webdriver.support import expected_conditions as EC
from selenium import webdriver
import logging
import urllib.parse
import atexit

# Configurar logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

app = Flask(__name__)

# --- Variável global para o ChromeDriver ---
DRIVER = None

# --- Função para configurar o Chrome em modo incógnito ---
def setup_chrome_driver():
    """Configura o Chrome em modo incógnito, bloqueando imagens e CSS."""
    global DRIVER
    if DRIVER is None:
        chrome_options = Options()
        chrome_options.add_argument("--incognito")  # Modo incógnito
        chrome_options.add_argument("--no-sandbox")
        chrome_options.add_argument("--disable-dev-shm-usage")
        # Bloquear imagens e CSS para acelerar
        chrome_options.add_argument("--blink-settings=imagesEnabled=false")
        chrome_options.add_experimental_option("prefs", {
            "profile.default_content_setting_values": {
                "images": 2,  # Bloqueia imagens
                "stylesheet": 2  # Bloqueia CSS
            }
        })
        try:
            # Tenta usar o ChromeDriver do sistema
            DRIVER = webdriver.Chrome(options=chrome_options)
            logger.info("ChromeDriver iniciado com sucesso.")
        except Exception as e:
            logger.error(f"Erro ao iniciar ChromeDriver: {str(e)}")
            raise Exception(f"Erro ao iniciar ChromeDriver. Verifique se o ChromeDriver está instalado e compatível com o Chrome (versão 140.0.7339.186). Detalhes: {str(e)}")
    return DRIVER

# --- Função para fechar o ChromeDriver ---
def close_chrome_driver():
    """Fecha o ChromeDriver ao encerrar o programa."""
    global DRIVER
    if DRIVER is not None:
        try:
            DRIVER.quit()
            logger.info("ChromeDriver fechado.")
        except Exception as e:
            logger.warning(f"Erro ao fechar ChromeDriver: {str(e)}")
        finally:
            DRIVER = None

# Registrar fechamento do ChromeDriver ao encerrar o programa
atexit.register(close_chrome_driver)

# --- Extrator com Selenium ---
def extrair_com_selenium(url: str):
    """Usa o Chrome em modo incógnito para extrair dados da página."""
    global DRIVER
    try:
        # Valida e corrige a URL
        parsed_url = urllib.parse.urlparse(url)
        if not parsed_url.scheme:
            url = f"https://{url}"
            logger.info(f"URL corrigida para: {url}")
        
        if DRIVER is None:
            DRIVER = setup_chrome_driver()
        
        logger.info(f"Abrindo URL com Selenium: {url}")
        DRIVER.get(url)
        
        # Espera a tag crucial carregar
        WebDriverWait(DRIVER, 30).until(
            EC.presence_of_element_located((By.ID, "initial-data"))
        )
        
        # Extrai o conteúdo da página
        page_source = DRIVER.page_source
        tree = html.fromstring(page_source)
        script_tag = tree.xpath("//script[@id='initial-data']")
        
        if not script_tag:
            logger.error("Tag <script id='initial-data'> não encontrada.")
            return {"error": "Tag <script id='initial-data'> não encontrada."}

        json_string = script_tag[0].get('data-json')
        if not json_string:
            logger.error("Atributo 'data-json' não encontrado na tag.")
            return {"error": "Atributo 'data-json' não encontrado na tag."}

        try:
            data = json.loads(json_string)
            logger.info("Dados extraídos com sucesso via Selenium.")
            return montar_dados(data)
        except json.JSONDecodeError as e:
            logger.error(f"Erro ao decodificar JSON: {str(e)}")
            return {"error": f"Erro ao decodificar JSON: {str(e)}"}
    
    except Exception as e:
        logger.error(f"Erro no Selenium: {str(e)}")
        return {"error": f"Erro ao usar Selenium: {str(e)}"}

# --- Mapeamento dos Dados ---
def montar_dados(data):
    """Mapeia os dados extraídos para o formato final."""
    ad_data = data.get('ad', {})
    
    return {
        'title': ad_data.get('subject', None),
        'price': ad_data.get('priceValue', None),
        'description': ad_data.get('body', '').replace('<br>', '\n'),
        'seller_name': ad_data.get('user', {}).get('name', None),
        'list_id': ad_data.get('adId', None),
        'ad_id': ad_data.get('listId', None),
        'publish_date': ad_data.get('listTime', None),
        'location': ad_data.get('locationProperties', {}),
        'properties': {
            prop.get('label'): prop.get('value')
            for prop in ad_data.get('properties', []) if prop.get('label') and prop.get('value')
        },
        'images': [
            img.get('original')
            for img in ad_data.get('images', []) if img.get('original')
        ]
    }

# --- Rota da API ---
@app.route('/extrair', methods=['GET'])
def api_endpoint():
    """Endpoint principal da API."""
    url_para_extrair = request.args.get('url')
    if not url_para_extrair:
        logger.warning("Parâmetro 'url' não fornecido.")
        return jsonify({"error": "Parâmetro 'url' é obrigatório."}), 400

    # Corrige a URL antes de processar
    parsed_url = urllib.parse.urlparse(url_para_extrair)
    if not parsed_url.scheme:
        url_para_extrair = f"https://{url_para_extrair}"
        logger.info(f"URL corrigida na entrada: {url_para_extrair}")

    # Tenta Cloudscraper primeiro para maior velocidade
    try:
        scraper = cloudscraper.create_scraper()
        logger.info(f"Tentando Cloudscraper para: {url_para_extrair}")
        response = scraper.get(url_para_extrair, timeout=15)
        response.raise_for_status()
        
        tree = html.fromstring(response.content)
        script_tag = tree.xpath("//script[@id='initial-data']")
        
        if script_tag and script_tag[0].get('data-json'):
            logger.info("Extração bem-sucedida com Cloudscraper.")
            json_string = script_tag[0].get('data-json')
            try:
                data = json.loads(json_string)
                return jsonify(montar_dados(data))
            except json.JSONDecodeError as e:
                logger.error(f"Erro ao decodificar JSON com Cloudscraper: {str(e)}")
                return jsonify({"error": f"Erro ao decodificar JSON: {str(e)}"}), 500
        else:
            logger.info("Cloudscraper falhou ou não encontrou a tag, tentando Selenium...")
            
    except Exception as e:
        logger.warning(f"Cloudscraper falhou: {str(e)}. Tentando Selenium...")

    # Fallback para Selenium
    dados = extrair_com_selenium(url_para_extrair)
    if "error" in dados:
        return jsonify(dados), 500

    return jsonify(dados)

# --- Inicialização ---
if __name__ == '__main__':
    PORTA = 7701
    logger.info(f"Servidor iniciado: http://127.0.0.1:{PORTA}/extrair?url=<sua_url>")
    app.run(host='0.0.0.0', port=PORTA, threaded=True)