﻿<?php

set_time_limit(300); // Aumenta o tempo limite para 300 segundos para evitar timeout

header("Content-Type: application/json; charset=UTF-8");

// Define o caminho para o arquivo do token
$tokenFile = 'token.json';

/**
 * Função para consultar uma API remota usando cURL.
 *
 * @param string $url O URL do endpoint da API.
 * @param array $headers Um array de cabeçalhos HTTP para enviar com a requisição.
 * @return array Um array associativo com os detalhes da resposta.
 */
function consultarAPI($url, $headers) {
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL => $url,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HTTPHEADER => $headers,
        CURLOPT_HEADER => true,
        CURLOPT_TIMEOUT => 60
    ]);

    $startTime = microtime(true); // Registra o tempo de início para debug
    $response = curl_exec($ch);
    $executionTime = microtime(true) - $startTime; // Calcula o tempo de execução
    $info = curl_getinfo($ch);
    $error = curl_error($ch);
    curl_close($ch);

    // Log do tempo de execução para debug
    error_log("API call to $url took $executionTime seconds");

    return [
        "rota" => $url,
        "status" => $info["http_code"] ?? 0,
        "online" => ($info["http_code"] >= 200 && $info["http_code"] < 500),
        "headers" => substr($response, 0, $info["header_size"] ?? 0),
        "body" => substr($response, $info["header_size"] ?? 0),
        "erroCurl" => $error
    ];
}

/**
 * Função para processar partes e formatar processo.
 *
 * @param array $tramitacaoAtual Dados da tramitação atual do processo.
 * @param string $cpfCnpj CPF/CNPJ para contagem (opcional, usado na consulta por CPF/CNPJ).
 * @param string $advogadoPrincipal Nome do advogado principal (opcional, usado na consulta por OAB).
 * @param string $oabNumero Número da OAB (opcional, usado na consulta por OAB).
 * @param string $oabUf UF da OAB (opcional, usado na consulta por OAB).
 * @param string $consultaTipo Tipo de consulta ('OAB' ou 'CPF').
 * @return array Dados formatados do processo e contagens.
 */
function processarPartes($tramitacaoAtual, $cpfCnpj = '', $advogadoPrincipal = '', $oabNumero = '', $oabUf = '', $consultaTipo = '') {
    $poloAtivo = [];
    $poloPassivo = [];
    $advogados = [];
    $contagemAtivo = 0;
    $contagemPassivo = 0;
    $advogadoNoPoloAtivo = false;
    $advogadoNoPoloPassivo = false;
    $advogadoComoParteAtiva = false;

    if (isset($tramitacaoAtual['partes']) && is_array($tramitacaoAtual['partes'])) {
        foreach ($tramitacaoAtual['partes'] as $parte) {
            if (!isset($parte['polo'], $parte['nome'])) continue;

            $nomeParte = $parte['nome'];
            $documentoParte = '';

            // Extrai CPF/CNPJ da parte
            if (isset($parte['documentosPrincipais']) && is_array($parte['documentosPrincipais'])) {
                foreach ($parte['documentosPrincipais'] as $doc) {
                    if (in_array($doc['tipo'], ['CPF', 'CNPJ'])) {
                        $documentoParte = $doc['numero'];
                        break;
                    }
                }
            }

            $stringParte = $nomeParte . ($documentoParte ? " (Doc: $documentoParte)" : "");

            // Agrupa por polo
            if ($parte['polo'] === 'ATIVO') {
                $poloAtivo[] = $stringParte;

                // Verifica contagem para CPF/CNPJ
                if ($consultaTipo === 'CPF' && $cpfCnpj && $documentoParte === $cpfCnpj) {
                    $contagemAtivo = 1;
                }

                // Verifica representantes no polo ativo
                if (isset($parte['representantes']) && is_array($parte['representantes'])) {
                    foreach ($parte['representantes'] as $representante) {
                        if (isset($representante['tipoRepresentacao']) && $representante['tipoRepresentacao'] === 'ADVOGADO') {
                            $nomeAdv = $representante['nome'] ?? 'N/A';
                            $cpfAdv = 'N/A';
                            if (isset($representante['cadastroReceitaFederal']) && is_array($representante['cadastroReceitaFederal'])) {
                                foreach ($representante['cadastroReceitaFederal'] as $docAdv) {
                                    if ($docAdv['tipo'] === 'CPF') {
                                        $cpfAdv = $docAdv['numero'];
                                        break;
                                    }
                                }
                            }
                            $advogados[] = "Advogado: $nomeAdv" . ($cpfAdv !== 'N/A' ? " (CPF: $cpfAdv)" : "");

                            // Verifica OAB para consulta por OAB
                            if ($consultaTipo === 'OAB' && isset($representante['oab']) && is_array($representante['oab'])) {
                                foreach ($representante['oab'] as $oabInfo) {
                                    if (isset($oabInfo['numero'], $oabInfo['uf']) && $oabInfo['numero'] == $oabNumero && $oabInfo['uf'] == $oabUf) {
                                        $advogadoNoPoloAtivo = true;
                                    }
                                }
                            }
                        }
                    }
                }
            } elseif ($parte['polo'] === 'PASSIVO') {
                $poloPassivo[] = $stringParte;

                // Verifica contagem para CPF/CNPJ
                if ($consultaTipo === 'CPF' && $cpfCnpj && $documentoParte === $cpfCnpj) {
                    $contagemPassivo = 1;
                }
            }
        }
    }

    // Verifica se o advogado principal aparece no polo passivo ou como parte no polo ativo
    if ($consultaTipo === 'OAB' && $advogadoPrincipal) {
        if (strpos(implode('; ', $poloPassivo), $advogadoPrincipal) !== false) {
            $advogadoNoPoloPassivo = true;
            $contagemPassivo = 1;
        }
        if (strpos(implode('; ', $poloAtivo), $advogadoPrincipal) !== false) {
            $advogadoComoParteAtiva = true;
        }
        if ($advogadoNoPoloAtivo && !$advogadoComoParteAtiva) {
            $contagemAtivo = 1;
        }
    }

    return [
        'poloAtivo' => implode('; ', $poloAtivo) ?: 'N/A',
        'poloPassivo' => implode('; ', $poloPassivo) ?: 'N/A',
        'advogados' => implode('; ', $advogados) ?: 'N/A',
        'contagemAtivo' => $contagemAtivo,
        'contagemPassivo' => $contagemPassivo,
        'advogadoNoPoloAtivo' => $advogadoNoPoloAtivo,
        'advogadoNoPoloPassivo' => $advogadoNoPoloPassivo,
        'advogadoComoParteAtiva' => $advogadoComoParteAtiva
    ];
}

/**
 * Função para formatar processo para a saída.
 *
 * @param array $p Dados do processo.
 * @param array $tramitacaoAtual Dados da tramitação atual.
 * @param array $partes Dados processados das partes.
 * @return array Processo formatado.
 */
function formatarProcesso($p, $tramitacaoAtual, $partes) {
    return [
        "Detalhes do Processo" => [
            "Número do Processo" => $p['numeroProcesso'] ?? 'N/A',
            "Tribunal" => $tramitacaoAtual['tribunal']['nome'] ?? 'N/A',
            "Órgão Julgador" => $tramitacaoAtual['orgaoJulgador']['nome'] ?? 'N/A',
            "Classe" => $tramitacaoAtual['classe'][0]['descricao'] ?? 'N/A',
            "Assunto" => $tramitacaoAtual['assunto'][0]['descricao'] ?? 'N/A',
            "Valor da Ação" => isset($tramitacaoAtual['valorAcao']) && is_numeric($tramitacaoAtual['valorAcao']) ? 'R$ ' . number_format($tramitacaoAtual['valorAcao'], 2, ',', '.') : 'R$ 0,00',
            "Data do Ajuizamento" => isset($tramitacaoAtual['dataHoraAjuizamento']) ? date('d/m/Y H:i', strtotime($tramitacaoAtual['dataHoraAjuizamento'])) : 'N/A',
            "Data do Último Movimento" => isset($tramitacaoAtual['ultimoMovimento']['dataHora']) ? date('d/m/Y H:i', strtotime($tramitacaoAtual['ultimoMovimento']['dataHora'])) : 'N/A'
        ],
        "Polo Ativo" => [
            "Parte" => $partes['poloAtivo'],
            "Advogado" => $partes['advogados']
        ],
        "Polo Passivo" => [
            "Parte" => $partes['poloPassivo']
        ]
    ];
}

// ============================
// GESTÃO DO TOKEN
// ============================
$bearerToken = "";
$headers = [];

// Verifica se um novo token foi passado via requisição GET
if (isset($_REQUEST['bearer'])) {
    $bearerToken = 'Bearer ' . filter_var($_REQUEST['bearer'], FILTER_SANITIZE_SPECIAL_CHARS);
    
    // Cria ou atualiza o arquivo token.json com o novo token
    $tokenData = [
        'token' => $bearerToken,
        'created_at' => date('Y-m-d H:i:s'),
        'expires_in' => '' // Este valor precisa ser obtido do payload do token
    ];
    file_put_contents($tokenFile, json_encode($tokenData, JSON_PRETTY_PRINT));
} else {
    // Lê o token do arquivo JSON
    if (file_exists($tokenFile)) {
        $tokenData = json_decode(file_get_contents($tokenFile), true);
        if (is_array($tokenData) && isset($tokenData['token'])) {
            $bearerToken = $tokenData['token'];
        }
    }
    // Se não houver token, retorna erro
    if (empty($bearerToken)) {
        echo json_encode([
            'error' => 'Nenhum token de autenticação fornecido ou encontrado.'
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }
}

// ============================
// CABEÇALHOS FIXOS (com o token)
// ============================
$headers = [
    "Accept: application/json, text/plain, */*",
    "Accept-Encoding: gzip, deflate, br, zstd",
    "Accept-Language: pt-BR,pt;q=0.9,en-US;q=0.8,en;q=0.7",
    "Authorization: " . $bearerToken,
    "Connection: keep-alive",
    "Referer: https://portaldeservicos.pdpj.jus.br/consulta",
    "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64)"
];

// ============================
// VALORES VINDO VIA GET/POST
// ============================
$oab = filter_var($_REQUEST["oab"] ?? "", FILTER_SANITIZE_SPECIAL_CHARS);
$processo = filter_var($_REQUEST["processo"] ?? "", FILTER_SANITIZE_SPECIAL_CHARS);
$cpf = filter_var($_REQUEST["cpf"] ?? "", FILTER_SANITIZE_SPECIAL_CHARS);

// Separa UF e número da OAB para verificação
$oabUf = substr($oab, 0, 2);
$oabNumero = substr($oab, 2);

// ============================
// ROTAS BASE
// ============================
$rotas = [
    "Consulta por OAB" => $oab ? "https://portaldeservicos.pdpj.jus.br/api/v2/processos?oabRepresentante=" . urlencode($oab) : "",
    "Consulta por Processo" => $processo ? "https://portaldeservicos.pdpj.jus.br/api/v2/processos/" . $processo : "",
    "Consulta por CPF/CNPJ" => $cpf ? "https://portaldeservicos.pdpj.jus.br/api/v2/processos?cpfCnpjParte=" . urlencode($cpf) : ""
];

// ============================
// EXECUTAR CONSULTAS E REFORMATAR A RESPOSTA
// ============================
$resultados = [];
$allProcesses = [];
$totalProcessos = 0;
$processosRetornados = 0;
$contagemAtivo = 0; // Contagem de processos no polo ativo
$contagemPassivo = 0; // Contagem de processos no polo passivo
$maxProcessos = 1000;
$pageCount = 0;
$maxPages = 10; // Limite de segurança para número de páginas
$advogadoNomes = []; // Array para contar frequência dos nomes dos advogados (para OAB)

// Determina qual consulta foi solicitada
$consultaSolicitada = "";
if (!empty($oab)) {
    $consultaSolicitada = "Consulta por OAB";
} elseif (!empty($processo)) {
    $consultaSolicitada = "Consulta por Processo";
} elseif (!empty($cpf)) {
    $consultaSolicitada = "Consulta por CPF/CNPJ";
}

// Executa a consulta solicitada
if ($consultaSolicitada && !empty($rotas[$consultaSolicitada])) {
    if ($consultaSolicitada === "Consulta por OAB" || $consultaSolicitada === "Consulta por CPF/CNPJ") {
        // Lógica de paginação para consulta por OAB ou CPF/CNPJ
        $url = $rotas[$consultaSolicitada];
        $searchAfter = null;

        do {
            $pageCount++;
            if ($pageCount > $maxPages) {
                error_log("Limite de páginas ($maxPages) atingido, interrompendo loop");
                break;
            }

            $currentUrl = $url;
            if ($searchAfter) {
                $currentUrl .= '&searchAfter=' . implode(',', $searchAfter);
            }

            $apiResponse = consultarAPI($currentUrl, $headers);

            // Verifica erro no cURL
            if (!empty($apiResponse['erroCurl']) || !$apiResponse['online']) {
                $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Erro na requisição: " . $apiResponse['erroCurl']]);
                break;
            } elseif (!empty($apiResponse["body"])) {
                // Decodifica o JSON da resposta
                $responseData = json_decode($apiResponse["body"], true);
                if (json_last_error() !== JSON_ERROR_NONE) {
                    $resultados[$consultaSolicitada] = array_merge($apiResponse, ["json_error" => json_last_error_msg()]);
                    break;
                }

                // Define o total de processos na primeira iteração
                if ($totalProcessos === 0) {
                    $totalProcessos = $responseData['total'] ?? 0;
                }

                // Adiciona os processos ativos ao array, limitando a 5000
                if (isset($responseData['content']) && is_array($responseData['content'])) {
                    foreach ($responseData['content'] as $p) {
                        // Verifica se o processo é ativo
                        $isActive = false;
                        $tramitacaoAtual = null;
                        if (isset($p['tramitacoes']) && is_array($p['tramitacoes'])) {
                            foreach ($p['tramitacoes'] as $tramitacao) {
                                if (isset($tramitacao['ativo']) && $tramitacao['ativo'] === true) {
                                    $isActive = true;
                                    $tramitacaoAtual = $tramitacao;
                                    break;
                                }
                            }
                        }
                        if ($isActive && $tramitacaoAtual) {
                            // Coleta nomes de advogados para consulta por OAB
                            if ($consultaSolicitada === "Consulta por OAB") {
                                if (isset($tramitacaoAtual['partes']) && is_array($tramitacaoAtual['partes'])) {
                                    foreach ($tramitacaoAtual['partes'] as $parte) {
                                        if (isset($parte['polo']) && $parte['polo'] === 'ATIVO' && isset($parte['representantes']) && is_array($parte['representantes'])) {
                                            foreach ($parte['representantes'] as $representante) {
                                                if (isset($representante['tipoRepresentacao']) && $representante['tipoRepresentacao'] === 'ADVOGADO' && isset($representante['oab']) && is_array($representante['oab'])) {
                                                    foreach ($representante['oab'] as $oabInfo) {
                                                        if (isset($oabInfo['numero'], $oabInfo['uf']) && $oabInfo['numero'] == $oabNumero && $oabInfo['uf'] == $oabUf) {
                                                            $nomeAdv = $representante['nome'] ?? 'N/A';
                                                            $advogadoNomes[$nomeAdv] = ($advogadoNomes[$nomeAdv] ?? 0) + 1;
                                                        }
                                                    }
                                                }
                                            }
                                        }
                                    }
                                }
                            }

                            // Determina o advogado principal para consulta por OAB
                            $advogadoPrincipal = '';
                            if ($consultaSolicitada === "Consulta por OAB" && !empty($advogadoNomes)) {
                                arsort($advogadoNomes);
                                $advogadoPrincipal = key($advogadoNomes);
                            }

                            // Processa partes
                            $partes = processarPartes(
                                $tramitacaoAtual,
                                $consultaSolicitada === "Consulta por CPF/CNPJ" ? $cpf : '',
                                $advogadoPrincipal,
                                $oabNumero,
                                $oabUf,
                                $consultaSolicitada === "Consulta por OAB" ? 'OAB' : ($consultaSolicitada === "Consulta por CPF/CNPJ" ? 'CPF' : '')
                            );

                            // Incrementa contagens
                            $contagemAtivo += $partes['contagemAtivo'];
                            $contagemPassivo += $partes['contagemPassivo'];

                            // Adiciona processo à lista, respeitando o limite
                            if ($processosRetornados < $maxProcessos) {
                                // Para OAB, só inclui se o advogado está no polo ativo como representante
                                if ($consultaSolicitada !== "Consulta por OAB" || 
                                    ($partes['advogadoNoPoloAtivo'] && !$partes['advogadoNoPoloPassivo'] && !$partes['advogadoComoParteAtiva'] && strpos($partes['advogados'], "Advogado: $advogadoPrincipal") !== false)) {
                                    // Formata o processo
                                    $processoFormatado = formatarProcesso($p, $tramitacaoAtual, $partes);
                                    $allProcesses[] = $processoFormatado;
                                    $processosRetornados++;
                                }
                            }
                        }
                    }
                }

                // Prepara o próximo searchAfter
                $searchAfter = isset($responseData['searchAfter']) ? $responseData['searchAfter'] : null;

                // Verifica se há mais páginas
                $hasMore = isset($responseData['numberOfElements']) && $responseData['numberOfElements'] > 0 && $searchAfter !== null;

            } else {
                $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Resposta vazia ou inválida"]);
                break;
            }
        } while ($hasMore);

        // Após coletar todos, retorna a lista de processos como array JSON organizado
        $resultados[$consultaSolicitada] = [
            "Retornou" => "$processosRetornados de $totalProcessos processos ($contagemAtivo ativo, $contagemPassivo passivo)",
            "Processos" => $allProcesses
        ];

    } elseif ($consultaSolicitada === "Consulta por Processo") {
        // Para consulta por processo individual
        $url = $rotas[$consultaSolicitada];
        $apiResponse = consultarAPI($url, $headers);
        
        // Verifica erro no cURL
        if (!empty($apiResponse['erroCurl']) || !$apiResponse['online']) {
            $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Erro na requisição: " . $apiResponse['erroCurl']]);
        } elseif (!empty($apiResponse["body"])) {
            // Decodifica o JSON da resposta
            $processoData = json_decode($apiResponse["body"], true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $resultados[$consultaSolicitada] = array_merge($apiResponse, ["json_error" => json_last_error_msg()]);
            } elseif (is_array($processoData) && !empty($processoData) && isset($processoData[0])) {
                // Supõe que a resposta é um array com um único objeto
                $p = $processoData[0];
                
                // Verifica se o processo é ativo e pega tramitacaoAtual
                $tramitacaoAtual = null;
                if (isset($p['tramitacaoAtual'])) {
                    $tramitacaoAtual = $p['tramitacaoAtual'];
                }
                if ($tramitacaoAtual) {
                    // Processa partes usando a função existente
                    $partes = processarPartes($tramitacaoAtual);

                    // Formata o processo usando a função existente
                    $reformattedData = formatarProcesso($p, $tramitacaoAtual, $partes);

                    // Combina os dados processados na resposta
                    $resultados[$consultaSolicitada] = [
                        "Dados Processados" => $reformattedData
                    ];
                } else {
                    $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Nenhuma tramitação ativa encontrada"]);
                }
            } else {
                $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Resposta vazia ou inválida"]);
            }
        } else {
            $resultados[$consultaSolicitada] = array_merge($apiResponse, ["error" => "Resposta vazia ou inválida"]);
        }
    }
}

// ============================
// RESPOSTA JSON
// ============================
$finalResponse = ['response' => $resultados[$consultaSolicitada] ?? ['error' => 'Nenhuma consulta válida fornecida']];
echo json_encode($finalResponse, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

?>